import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Button, Form, Accordion, Badge, Alert } from 'react-bootstrap';
import ReactQuill from 'react-quill';
import 'react-quill/dist/quill.snow.css';
import { useDispatch, useSelector } from 'react-redux';
import { useNavigate } from 'react-router-dom';
import { PageMetaSection, GetBlogPageBanner, GetBlogPosts, GetBlogPostsListing } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { formatDateForUI } from '../../../data/global';
import axios from 'axios';
import './BlogsPage.css';

const BlogsPage = () => {
  const dispatch = useDispatch();
  const navigate = useNavigate();
  const { blogPostsListing, isLoading, blogPageBanner, pageMeta } = useSelector((state) => state.PropertyMangementReducer);
  console.log('[blogPostsListing]', blogPostsListing)
  console.log('[blogPageBanner]', blogPageBanner)
  console.log('[pageMeta]', pageMeta)
  // Meta form state
  const [metaForm, setMetaForm] = useState({
    meta_title: '',
    meta_description: '',
    meta_keywords: [],
    slug: '',
    canonical_tag: '',
    og_url: '',
    og_sitename: '',
    page_for: ''
  });
  const [metaKeywordInput, setMetaKeywordInput] = useState('');
  const [metaLoading, setMetaLoading] = useState(false);
  const [metaMessage, setMetaMessage] = useState({ type: '', text: '' });

  // Banner form state
  const [bannerForm, setBannerForm] = useState({
    main_heading: '',
    content: '',
    blog_image: null
  });
  const [bannerImagePreview, setBannerImagePreview] = useState(null);
  const [bannerLoading, setBannerLoading] = useState(false);
  const [bannerMessage, setBannerMessage] = useState({ type: '', text: '' });

  // Blog posts state
  const [selectedBlog, setSelectedBlog] = useState(null);
  const [blogLoading, setBlogLoading] = useState(false);

  useEffect(() => {
    dispatch(GetBlogPostsListing());
    dispatch(GetBlogPageBanner());
    dispatch(PageMetaSection("blogs_page"));
  }, [dispatch]);

  // Prefill meta form when data is available
  useEffect(() => {
    if (pageMeta && pageMeta.id) {
      // Parse meta_keywords from JSON string to array
      let keywords = [];
      try {
        keywords = pageMeta.meta_keywords ? JSON.parse(pageMeta.meta_keywords) : [];
      } catch (error) {
        console.error('Error parsing meta_keywords:', error);
        keywords = [];
      }

      // Ensure keywords is always an array
      if (!Array.isArray(keywords)) {
        keywords = [];
      }

      setMetaForm({
        meta_title: pageMeta.meta_title || '',
        meta_description: pageMeta.meta_description || '',
        meta_keywords: keywords,
        slug: pageMeta.slug || '',
        canonical_tag: pageMeta.canonical_tag || '',
        og_url: pageMeta.og_url || '',
        og_sitename: pageMeta.og_sitename || '',
        page_for: pageMeta.page_for || ''
      });
    }
  }, [pageMeta]);

  // Prefill banner form when data is available
  useEffect(() => {
    if (blogPageBanner && blogPageBanner.length > 0) {
      const existingBanner = blogPageBanner[0];
      setBannerForm({
        main_heading: existingBanner.main_heading || '',
        content: existingBanner.content || '',
        blog_image: null
      });

      // Set image preview if available
      if (existingBanner.blog_image) {
        setBannerImagePreview(existingBanner.blog_image);
      }
    }
  }, [blogPageBanner]);

  // Quill editor configuration
  const quillModules = {
    toolbar: [
      [{ 'header': [1, 2, 3, false] }],
      ['bold', 'italic', 'underline', 'strike'],
      [{ 'list': 'ordered' }, { 'list': 'bullet' }],
      [{ 'color': [] }, { 'background': [] }],
      [{ 'align': [] }],
      ['link', 'image'],
      ['clean']
    ],
  };

  const quillFormats = [
    'header',
    'bold', 'italic', 'underline', 'strike',
    'list', 'bullet',
    'color', 'background',
    'align',
    'link', 'image'
  ];

  // Page options for dropdown
  const pageOptions = [
    {
      label: "Home page",
      value: "home_page",
    },
    {
      label: "Blogs page",
      value: "blogs_page",
    },
    {
      label: "Service page",
      value: "services_page",
    }
  ];

  // Meta form handlers
  const handleMetaInputChange = (e) => {
    const { name, value } = e.target;
    setMetaForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleMetaKeywordInputChange = (e) => {
    setMetaKeywordInput(e.target.value);
  };

  const handleMetaKeywordKeyPress = (e) => {
    if (e.key === 'Enter' && metaKeywordInput.trim()) {
      e.preventDefault();
      const keyword = metaKeywordInput.trim();

      // Ensure meta_keywords is always an array
      const currentKeywords = Array.isArray(metaForm.meta_keywords) ? metaForm.meta_keywords : [];

      if (!currentKeywords.includes(keyword)) {
        setMetaForm(prev => ({
          ...prev,
          meta_keywords: [...currentKeywords, keyword]
        }));
      }
      setMetaKeywordInput('');
    }
  };

  const removeMetaKeyword = (keywordToRemove) => {
    setMetaForm(prev => ({
      ...prev,
      meta_keywords: Array.isArray(prev.meta_keywords)
        ? prev.meta_keywords.filter(keyword => keyword !== keywordToRemove)
        : []
    }));
  };

  const handleMetaSubmit = async (e) => {
    e.preventDefault();
    setMetaLoading(true);
    setMetaMessage({ type: '', text: '' });

    try {
      let response;
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'application/json'
        }
      };

      // Prepare form data with meta_keywords as JSON string
      const formDataToSend = {
        ...metaForm,
        meta_keywords: metaForm.meta_keywords
      };

      console.log('metaForm.meta_keywords', metaForm.meta_keywords)


      // Check if meta record already exists
      if (pageMeta && pageMeta.id) {
        // Update existing record - use PATCH
        response = await axios.patch(`http://localhost:3000/api/meta/${pageMeta.id}`, formDataToSend, config);
        if (response.status === 200) {
          setMetaMessage({ type: 'success', text: 'Meta information updated successfully!' });
        }
      } else {
        // Create new record - use POST
        response = await axios.post('http://localhost:3000/api/meta', formDataToSend, config);
        if (response.status === 201) {
          setMetaMessage({ type: 'success', text: 'Meta information created successfully!' });
        }
      }

      // Refresh the meta data after successful submission
      if (response.status === 200 || response.status === 201) {
        dispatch(PageMetaSection("blogs_page"));
      }
    } catch (error) {
      console.error('Error:', error);
      setMetaMessage({
        type: 'error',
        text: error.response?.data?.message || 'Error updating meta information'
      });
    } finally {
      setMetaLoading(false);
    }
  };

  // Banner form handlers
  const handleBannerInputChange = (e) => {
    const { name, value } = e.target;
    setBannerForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleBannerContentChange = (content) => {
    setBannerForm(prev => ({
      ...prev,
      content
    }));
  };

  const handleBannerImageChange = (e) => {
    const file = e.target.files[0];
    if (file) {
      setBannerForm(prev => ({
        ...prev,
        blog_image: file
      }));

      const reader = new FileReader();
      reader.onload = (e) => {
        setBannerImagePreview(e.target.result);
      };
      reader.readAsDataURL(file);
    }
  };

  const handleBannerSubmit = async (e) => {
    e.preventDefault();
    setBannerLoading(true);
    setBannerMessage({ type: '', text: '' });

    const formData = new FormData();
    formData.append('main_heading', bannerForm.main_heading);
    formData.append('content', bannerForm.content);

    if (bannerForm.blog_image) {
      formData.append('blog_image', bannerForm.blog_image);
    }

    try {
      let response;
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'multipart/form-data'
        }
      };

      // Check if banner record already exists
      if (blogPageBanner && blogPageBanner.length > 0) {
        // Update existing record - use PATCH
        const existingBanner = blogPageBanner[0]; // Assuming first record
        response = await axios.patch(`http://localhost:3000/api/page-blog/${existingBanner.id}`, formData, config);
        if (response.status === 200) {
          setBannerMessage({ type: 'success', text: 'Blog page banner updated successfully!' });
        }
      } else {
        // Create new record - use POST
        response = await axios.post('http://localhost:3000/api/page-blog', formData, config);
        if (response.status === 201) {
          setBannerMessage({ type: 'success', text: 'Blog page banner created successfully!' });
        }
      }

      // Refresh the banner data after successful submission
      if (response.status === 200 || response.status === 201) {
        dispatch(GetBlogPageBanner());
        // Reset form after successful submission
        setBannerForm({
          main_heading: '',
          content: '',
          blog_image: null
        });
        setBannerImagePreview(null);
      }
    } catch (error) {
      console.error('Error:', error);
      setBannerMessage({
        type: 'error',
        text: error.response?.data?.message || 'Error updating blog page banner'
      });
    } finally {
      setBannerLoading(false);
    }
  };

  // Blog post handlers
  const handleBlogClick = (blog) => {
    setSelectedBlog(blog);
    navigate(`/dashboard/blog-post/edit/${blog.id}`);
  };

  const getStatusBadge = (isActive, isFeatured) => {
    if (isActive && isFeatured) {
      return <Badge bg="success">Active & Featured</Badge>;
    } else if (isActive) {
      return <Badge bg="primary">Active</Badge>;
    } else if (isFeatured) {
      return <Badge bg="warning">Featured</Badge>;
    } else {
      return <Badge bg="secondary">Inactive</Badge>;
    }
  };

  return (
    <Container fluid className="blogs-page">
      <Row>
        <Col>
          <div className="page-header mb-4">
            <h2 className="mb-0">Blog Page Management</h2>
            <p className="text-muted" style={{ marginTop: '10px' }}>Manage your blog page banner and view all blog posts</p>
          </div>

          <Accordion defaultActiveKey="0" className="mb-4">
            {/* Meta Section */}
            <Accordion.Item eventKey="0">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-tags me-2"></i>
                  <span>Meta Information</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    {metaMessage.text && (
                      <Alert variant={metaMessage.type === 'success' ? 'success' : 'danger'} dismissible>
                        {metaMessage.text}
                      </Alert>
                    )}

                    <Form onSubmit={handleMetaSubmit}>
                      <Row>
                        <Col md={6}>
                          {/* Meta Title */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Meta Title *</Form.Label>
                            <Form.Control
                              type="text"
                              name="meta_title"
                              value={metaForm.meta_title}
                              onChange={handleMetaInputChange}
                              placeholder="Enter meta title"
                              required
                            />
                          </div>

                          {/* Meta Description */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Meta Description *</Form.Label>
                            <Form.Control
                              as="textarea"
                              rows={3}
                              name="meta_description"
                              value={metaForm.meta_description}
                              onChange={handleMetaInputChange}
                              placeholder="Enter meta description"
                              required
                            />
                          </div>

                          {/* Meta Keywords */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Meta Keywords</Form.Label>
                            <Form.Control
                              type="text"
                              value={metaKeywordInput}
                              onChange={handleMetaKeywordInputChange}
                              onKeyPress={handleMetaKeywordKeyPress}
                              placeholder="Type keyword and press Enter"
                            />
                            {Array.isArray(metaForm.meta_keywords) && metaForm.meta_keywords.length > 0 && (
                              <div className="keywords-tags mt-2">
                                {metaForm.meta_keywords.map((keyword, index) => (
                                  <Badge
                                    key={index}
                                    bg="primary"
                                    className="me-2 mb-2 keyword-tag"
                                  >
                                    {keyword}
                                    <button
                                      type="button"
                                      className="btn-close btn-close-white ms-2"
                                      onClick={() => removeMetaKeyword(keyword)}
                                      style={{ fontSize: '8px' }}
                                    ></button>
                                  </Badge>
                                ))}
                              </div>
                            )}
                          </div>

                          {/* Slug */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Slug *</Form.Label>
                            <Form.Control
                              type="text"
                              name="slug"
                              value={metaForm.slug}
                              onChange={handleMetaInputChange}
                              placeholder="Enter slug"
                              required
                            />
                          </div>
                        </Col>

                        <Col md={6}>
                          {/* Canonical Tag */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Canonical Tag</Form.Label>
                            <Form.Control
                              type="url"
                              name="canonical_tag"
                              value={metaForm.canonical_tag}
                              onChange={handleMetaInputChange}
                              placeholder="https://example.com/page"
                            />
                          </div>

                          {/* OG URL */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">OG URL</Form.Label>
                            <Form.Control
                              type="url"
                              name="og_url"
                              value={metaForm.og_url}
                              onChange={handleMetaInputChange}
                              placeholder="https://example.com/page"
                            />
                          </div>

                          {/* OG Site Name */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">OG Site Name</Form.Label>
                            <Form.Control
                              type="text"
                              name="og_sitename"
                              value={metaForm.og_sitename}
                              onChange={handleMetaInputChange}
                              placeholder="Enter site name"
                            />
                          </div>

                          {/* Page For */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Page For *</Form.Label>
                            <Form.Select
                              name="page_for"
                              value={metaForm.page_for}
                              onChange={handleMetaInputChange}
                              required
                            >
                              <option value="">Select a page</option>
                              {pageOptions.map((option) => (
                                <option key={option.value} value={option.value}>
                                  {option.label}
                                </option>
                              ))}
                            </Form.Select>
                          </div>
                        </Col>
                      </Row>

                      <div className="d-flex justify-content-end">
                        <Button
                          variant="primary"
                          type="submit"
                          disabled={metaLoading}
                        >
                          {metaLoading ? 'Processing...' : (pageMeta && pageMeta.id ? 'Update Meta Information' : 'Create Meta Information')}
                        </Button>
                      </div>
                    </Form>
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* Banner Section */}
            <Accordion.Item eventKey="1">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-image me-2"></i>
                  <span>Blog Page Banner</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    {bannerMessage.text && (
                      <Alert variant={bannerMessage.type === 'success' ? 'success' : 'danger'} dismissible>
                        {bannerMessage.text}
                      </Alert>
                    )}

                    <Form onSubmit={handleBannerSubmit}>
                      <Row>
                        <Col md={8}>
                          {/* Main Heading */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Main Heading *</Form.Label>
                            <Form.Control
                              type="text"
                              name="main_heading"
                              value={bannerForm.main_heading}
                              onChange={handleBannerInputChange}
                              placeholder="Enter main banner heading"
                              required
                            />
                          </div>

                          {/* Content Editor */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Banner Content *</Form.Label>
                            <div className="quill-editor-container">
                              <ReactQuill
                                theme="snow"
                                value={bannerForm.content}
                                onChange={handleBannerContentChange}
                                modules={quillModules}
                                formats={quillFormats}
                                placeholder="Enter banner content..."
                                style={{ height: '200px' }}
                              />
                            </div>
                          </div>
                        </Col>

                        <Col md={4}>
                          {/* Banner Image */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Banner Background Image</Form.Label>
                            <div className="image-upload-section">
                              <Form.Control
                                type="file"
                                accept="image/*"
                                onChange={handleBannerImageChange}
                                className="mb-2"
                              />
                              {bannerImagePreview && (
                                <div className="image-preview">
                                  <img
                                    src={bannerImagePreview}
                                    alt="Banner Preview"
                                    className="img-thumbnail"
                                    style={{ maxWidth: '100%', height: '150px', objectFit: 'cover' }}
                                  />
                                </div>
                              )}
                            </div>
                          </div>
                        </Col>
                      </Row>

                      <div className="d-flex justify-content-end">
                        <Button
                          variant="primary"
                          type="submit"
                          disabled={bannerLoading}
                        >
                          {bannerLoading ? 'Processing...' : (blogPageBanner && blogPageBanner.length > 0 ? 'Update Banner' : 'Create Banner')}
                        </Button>
                      </div>
                    </Form>
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* Blog Posts Section */}
            <Accordion.Item eventKey="2">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-newspaper me-2"></i>
                  <span>Blog Posts ({blogPostsListing?.length || 0})</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    <div className="d-flex justify-content-between align-items-center mb-3">
                      <h5 className="mb-0">All Blog Posts</h5>
                      <Button
                        variant="primary"
                        onClick={() => navigate('/dashboard/blog-post/add')}
                      >
                        <i className="fas fa-plus me-2"></i>
                        Add New Blog Post
                      </Button>
                    </div>

                    {isLoading ? (
                      <div className="text-center py-5">
                        <div className="spinner-border text-primary" role="status">
                          <span className="visually-hidden">Loading...</span>
                        </div>
                        <p className="mt-2">Loading blog posts...</p>
                      </div>
                    ) : blogPostsListing && blogPostsListing.length > 0 ? (
                      <div className="blog-posts-grid">
                        {blogPostsListing.map((blog, index) => (
                          <Card
                            key={blog.id || index}
                            className="blog-post-card mb-3"
                            onClick={() => handleBlogClick(blog)}
                          >
                            <Card.Body>
                              <div className="d-flex justify-content-between align-items-start mb-2">
                                <h6 className="card-title mb-1">{blog.title}</h6>
                                {getStatusBadge(blog.is_active, blog.is_featured)}
                              </div>

                              <p className="text-muted small mb-2">
                                <strong>Slug:</strong> {blog.slug}
                              </p>

                              <p className="text-muted small mb-2">
                                <strong>Category:</strong> {blog.category_details?.name || 'N/A'}
                              </p>

                              <p className="text-muted small mb-0">
                                <strong>Created:</strong> {formatDateForUI(blog.createdAt)}
                              </p>
                            </Card.Body>

                            <Card.Footer className="bg-light">
                              <div className="d-flex justify-content-between align-items-center">
                                <small className="text-muted">
                                  Click to edit this blog post
                                </small>
                                <Button variant="outline-primary" size="sm">
                                  <i className="fas fa-edit me-1"></i>
                                  Edit
                                </Button>
                              </div>
                            </Card.Footer>
                          </Card>
                        ))}
                      </div>
                    ) : (
                      <div className="text-center py-5">
                        <i className="fas fa-newspaper fa-3x text-muted mb-3"></i>
                        <h5 className="text-muted">No Blog Posts Found</h5>
                        <p className="text-muted">Start by creating your first blog post.</p>
                        <Button
                          variant="primary"
                          onClick={() => navigate('/dashboard/blog-post/add')}
                        >
                          <i className="fas fa-plus me-2"></i>
                          Create First Blog Post
                        </Button>
                      </div>
                    )}
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>
          </Accordion>
        </Col>
      </Row>
    </Container>
  );
};

export default BlogsPage;