import React, { useState, useEffect } from 'react';
import { Container, Row, Col, Card, Button, Form, Accordion, Alert, Badge } from 'react-bootstrap';
import { useDispatch, useSelector } from 'react-redux';
import { useNavigate } from 'react-router-dom';
import { PageMetaSection, GetHomePageBanners, GetHomePageBannerJson, GetAllServicesJson, GetAchievementsJson, GetPromotionsJson, GetAllServicesTabsJson, GetPortfolioListingJson, GetReviewsJson, GetAllFaqsJson } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { formatDateForUI } from '../../../data/global';
import axios from 'axios';
import './BlogsPage.css';
import ReactQuill from 'react-quill';

const HomePage = () => {
  const dispatch = useDispatch();
  const navigate = useNavigate();
  const { pageMeta, homePageBannerJson, servicesJson, achievementsJson, promotionsJson, servicesTabsJson, portfolioListingJson, reviewsJson, faqsJson, isLoading } = useSelector((state) => state.PropertyMangementReducer);
  console.log('[pageMeta]', pageMeta)
  console.log('[homePageBannerJson]', homePageBannerJson)
  console.log('[servicesJson]', servicesJson)

  // Meta form state
  const [metaForm, setMetaForm] = useState({
    meta_title: '',
    meta_description: '',
    meta_keywords: [],
    slug: '',
    canonical_tag: '',
    og_url: '',
    og_sitename: '',
    page_for: ''
  });
  const [metaKeywordInput, setMetaKeywordInput] = useState('');
  const [metaLoading, setMetaLoading] = useState(false);
  const [metaMessage, setMetaMessage] = useState({ type: '', text: '' });

  // Services section form state
  const [servicesSectionForm, setServicesSectionForm] = useState({
    main_heading: ''
  });
  const [servicesSectionLoading, setServicesSectionLoading] = useState(false);
  const [servicesSectionMessage, setServicesSectionMessage] = useState({ type: '', text: '' });
  const [servicesSectionData, setServicesSectionData] = useState(null);

  // About section form state
  const [aboutSectionForm, setAboutSectionForm] = useState({
    span_heading: '',
    main_heading_1: '',
    main_heading_2: '',
    content: '',
    button_text: '',
    button_link: '',
    main_heading_img: null
  });
  const [aboutSectionLoading, setAboutSectionLoading] = useState(false);
  const [aboutSectionMessage, setAboutSectionMessage] = useState({ type: '', text: '' });
  const [aboutSectionData, setAboutSectionData] = useState(null);
  const [aboutImagePreview, setAboutImagePreview] = useState(null);
  const [aboutFileName, setAboutFileName] = useState('No file chosen');

  // Video section form state
  const [videoSectionForm, setVideoSectionForm] = useState({
    video: null,
    video_alt: ''
  });
  const [videoSectionLoading, setVideoSectionLoading] = useState(false);
  const [videoSectionMessage, setVideoSectionMessage] = useState({ type: '', text: '' });
  const [videoSectionData, setVideoSectionData] = useState(null);
  const [videoFileName, setVideoFileName] = useState('No file chosen');

  // Achievements section form state
  const [achievementsSectionForm, setAchievementsSectionForm] = useState({
    main_heading_1: '',
    main_heading_2: '',
    content_para_1: '',
    content_para_2: ''
  });
  const [achievementsSectionLoading, setAchievementsSectionLoading] = useState(false);
  const [achievementsSectionMessage, setAchievementsSectionMessage] = useState({ type: '', text: '' });
  const [achievementsSectionData, setAchievementsSectionData] = useState(null);

  // Quill editor configuration
  const quillModules = {
    toolbar: [
      [{ 'header': [1, 2, 3, false] }],
      ['bold', 'italic', 'underline', 'strike'],
      [{ 'list': 'ordered' }, { 'list': 'bullet' }],
      [{ 'color': [] }, { 'background': [] }],
      [{ 'align': [] }],
      ['link', 'image'],
      ['clean']
    ],
  };

  const quillFormats = [
    'header',
    'bold', 'italic', 'underline', 'strike',
    'list', 'bullet',
    'color', 'background',
    'align',
    'link', 'image'
  ];

  useEffect(() => {
    dispatch(PageMetaSection("home_page"));
    dispatch(GetHomePageBannerJson());
    dispatch(GetAllServicesJson());
    dispatch(GetAchievementsJson());
    dispatch(GetPromotionsJson());
    dispatch(GetAllServicesTabsJson());
    dispatch(GetPortfolioListingJson());
    dispatch(GetReviewsJson());
    dispatch(GetAllFaqsJson());
    fetchServicesSection();
    fetchAboutSection();
    fetchVideoSection();
    fetchAchievementsSection();
  }, [dispatch]);

  // Prefill meta form when data is available
  useEffect(() => {
    if (pageMeta && pageMeta.id) {
      // Parse meta_keywords from JSON string to array
      let keywords = [];
      try {
        keywords = pageMeta.meta_keywords ? JSON.parse(pageMeta.meta_keywords) : [];
      } catch (error) {
        console.error('Error parsing meta_keywords:', error);
        keywords = [];
      }

      // Ensure keywords is always an array
      if (!Array.isArray(keywords)) {
        keywords = [];
      }

      setMetaForm({
        meta_title: pageMeta.meta_title || '',
        meta_description: pageMeta.meta_description || '',
        meta_keywords: keywords,
        slug: pageMeta.slug || '',
        canonical_tag: pageMeta.canonical_tag || '',
        og_url: pageMeta.og_url || '',
        og_sitename: pageMeta.og_sitename || '',
        page_for: pageMeta.page_for || ''
      });
    }
  }, [pageMeta]);

  // Prefill services section form when data is available
  useEffect(() => {
    if (servicesSectionData && servicesSectionData.length > 0) {
      const existingSection = servicesSectionData[0];
      setServicesSectionForm({
        main_heading: existingSection.main_heading || ''
      });
    }
  }, [servicesSectionData]);

  // Prefill about section form when data is available
  useEffect(() => {
    if (aboutSectionData && aboutSectionData.length > 0) {
      const existingSection = aboutSectionData[0];
      setAboutSectionForm({
        span_heading: existingSection.span_heading || '',
        main_heading_1: existingSection.main_heading_1 || '',
        main_heading_2: existingSection.main_heading_2 || '',
        content: existingSection.content || '',
        button_text: existingSection.button_text || '',
        button_link: existingSection.button_link || '',
        main_heading_img: null
      });

      // Set image preview if available
      if (existingSection.main_heading_img) {
        setAboutImagePreview(existingSection.main_heading_img);
        setAboutFileName('Current image');
      }
    }
  }, [aboutSectionData]);

  // Prefill video section form when data is available
  useEffect(() => {
    if (videoSectionData && videoSectionData.length > 0) {
      const existingSection = videoSectionData[0];
      setVideoSectionForm({
        video: null,
        video_alt: existingSection.video_alt || ''
      });

      // Set file name if video exists
      if (existingSection.video) {
        setVideoFileName('Current video');
      }
    }
  }, [videoSectionData]);

  // Prefill achievements section form when data is available
  useEffect(() => {
    if (achievementsSectionData && achievementsSectionData.length > 0) {
      const existingSection = achievementsSectionData[0];
      setAchievementsSectionForm({
        main_heading_1: existingSection.main_heading_1 || '',
        main_heading_2: existingSection.main_heading_2 || '',
        content_para_1: existingSection.content_para_1 || '',
        content_para_2: existingSection.content_para_2 || ''
      });
    }
  }, [achievementsSectionData]);



  // Page options for dropdown
  const pageOptions = [
    {
      label: "Home page",
      value: "home_page",
    },
    {
      label: "Blogs page",
      value: "blogs_page",
    },
    {
      label: "Service page",
      value: "services_page",
    }
  ];

  // Meta form handlers
  const handleMetaInputChange = (e) => {
    const { name, value } = e.target;
    setMetaForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleMetaKeywordInputChange = (e) => {
    setMetaKeywordInput(e.target.value);
  };

  const handleMetaKeywordKeyPress = (e) => {
    if (e.key === 'Enter' && metaKeywordInput.trim()) {
      e.preventDefault();
      const keyword = metaKeywordInput.trim();

      // Ensure meta_keywords is always an array
      const currentKeywords = Array.isArray(metaForm.meta_keywords) ? metaForm.meta_keywords : [];

      if (!currentKeywords.includes(keyword)) {
        setMetaForm(prev => ({
          ...prev,
          meta_keywords: [...currentKeywords, keyword]
        }));
      }
      setMetaKeywordInput('');
    }
  };

  const removeMetaKeyword = (keywordToRemove) => {
    setMetaForm(prev => ({
      ...prev,
      meta_keywords: Array.isArray(prev.meta_keywords)
        ? prev.meta_keywords.filter(keyword => keyword !== keywordToRemove)
        : []
    }));
  };

  const handleMetaSubmit = async (e) => {
    e.preventDefault();
    setMetaLoading(true);
    setMetaMessage({ type: '', text: '' });

    try {
      let response;
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'application/json'
        }
      };

      // Prepare form data with meta_keywords as JSON string
      const formDataToSend = {
        ...metaForm,
        meta_keywords: metaForm.meta_keywords
      };

      console.log('metaForm.meta_keywords', metaForm.meta_keywords)

      // Check if meta record already exists
      if (pageMeta && pageMeta.id) {
        // Update existing record - use PATCH
        response = await axios.patch(`http://localhost:3000/api/meta/${pageMeta.id}`, formDataToSend, config);
        if (response.status === 200) {
          setMetaMessage({ type: 'success', text: 'Meta information updated successfully!' });
        }
      } else {
        // Create new record - use POST
        response = await axios.post('http://localhost:3000/api/meta', formDataToSend, config);
        if (response.status === 201) {
          setMetaMessage({ type: 'success', text: 'Meta information created successfully!' });
        }
      }

      // Refresh the meta data after successful submission
      if (response.status === 200 || response.status === 201) {
        dispatch(PageMetaSection("home_page"));
      }
    } catch (error) {
      console.error('Error:', error);
      setMetaMessage({
        type: 'error',
        text: error.response?.data?.message || 'Error updating meta information'
      });
    } finally {
      setMetaLoading(false);
    }
  };

  // Banner handlers
  const handleBannerClick = (banner) => {
    navigate('/dashboard/banners');
  };

  // Services section handlers
  const fetchServicesSection = async () => {
    try {
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'application/json'
        }
      };
      const response = await axios.get('http://localhost:3000/api/home/services-section', config);
      if (response.data.success) {
        setServicesSectionData(response.data.data);
      }
    } catch (error) {
      console.error('Error fetching services section:', error);
    }
  };

  const handleServicesSectionInputChange = (e) => {
    const { name, value } = e.target;
    setServicesSectionForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleServicesSectionSubmit = async (e) => {
    e.preventDefault();
    setServicesSectionLoading(true);
    setServicesSectionMessage({ type: '', text: '' });

    try {
      let response;
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'application/json'
        }
      };

      // Check if services section record already exists
      if (servicesSectionData && servicesSectionData.length > 0) {
        // Update existing record - use PATCH
        const existingSection = servicesSectionData[0];
        response = await axios.patch(`http://localhost:3000/api/home/services-section/${existingSection.id}`, servicesSectionForm, config);
        if (response.status === 200) {
          setServicesSectionMessage({ type: 'success', text: 'Services section heading updated successfully!' });
        }
      } else {
        // Create new record - use POST
        response = await axios.post('http://localhost:3000/api/home/services-section', servicesSectionForm, config);
        if (response.status === 201) {
          setServicesSectionMessage({ type: 'success', text: 'Services section heading created successfully!' });
        }
      }

      // Refresh the services section data after successful submission
      if (response.status === 200 || response.status === 201) {
        fetchServicesSection();
      }
    } catch (error) {
      console.error('Error:', error);
      setServicesSectionMessage({
        type: 'error',
        text: error.response?.data?.message || 'Error updating services section'
      });
    } finally {
      setServicesSectionLoading(false);
    }
  };

  // Service handlers
  const handleServiceClick = (service) => {
    navigate('/dashboard/services');
  };

  const handleAchievementClick = (achievement) => {
    navigate('/dashboard/achievements');
  };

  const handlePromotionClick = (promotion) => {
    navigate('/dashboard/promotions');
  };

      const handleServiceTabClick = (serviceTab) => {
      navigate('/dashboard/services-tabs');
    };

    const handlePortfolioClick = (portfolio) => {
      navigate('/dashboard/portfolio');
    };

    const handleReviewClick = (review) => {
      navigate('/dashboard/reviews');
    };

    const handleFaqClick = (faq) => {
      navigate('/dashboard/faqs');
    };

  // About section handlers
  const fetchAboutSection = async () => {
    try {
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'application/json'
        }
      };
      const response = await axios.get('http://localhost:3000/api/home/who-we-are', config);
      if (response.data.success) {
        setAboutSectionData(response.data.data);
      }
    } catch (error) {
      console.error('Error fetching about section:', error);
    }
  };

  const handleAboutSectionInputChange = (e) => {
    const { name, value } = e.target;
    setAboutSectionForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleAboutContentChange = (content) => {
    setAboutSectionForm(prev => ({
      ...prev,
      content
    }));
  };

  const handleAboutImageChange = (e) => {
    const file = e.target.files[0];
    if (file) {
      setAboutSectionForm(prev => ({
        ...prev,
        main_heading_img: file
      }));
      setAboutFileName(file.name);

      const reader = new FileReader();
      reader.onload = (e) => {
        setAboutImagePreview(e.target.result);
      };
      reader.readAsDataURL(file);
    }
  };

  const handleAboutSectionSubmit = async (e) => {
    e.preventDefault();
    setAboutSectionLoading(true);
    setAboutSectionMessage({ type: '', text: '' });

    try {
      const formData = new FormData();
      formData.append('span_heading', aboutSectionForm.span_heading);
      formData.append('main_heading_1', aboutSectionForm.main_heading_1);
      formData.append('main_heading_2', aboutSectionForm.main_heading_2);
      formData.append('content', aboutSectionForm.content);
      formData.append('button_text', aboutSectionForm.button_text);
      formData.append('button_link', aboutSectionForm.button_link);

      if (aboutSectionForm.main_heading_img) {
        formData.append('main_heading_img', aboutSectionForm.main_heading_img);
      }

      let response;
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'multipart/form-data'
        }
      };

      // Check if about section record already exists
      if (aboutSectionData && aboutSectionData.length > 0) {
        // Update existing record - use PATCH
        const existingSection = aboutSectionData[0];
        response = await axios.patch(`http://localhost:3000/api/home/who-we-are/${existingSection.id}`, formData, config);
        if (response.status === 200) {
          setAboutSectionMessage({ type: 'success', text: 'About section updated successfully!' });
        }
      } else {
        // Create new record - use POST
        response = await axios.post('http://localhost:3000/api/home/who-we-are', formData, config);
        if (response.status === 201) {
          setAboutSectionMessage({ type: 'success', text: 'About section created successfully!' });
        }
      }

      // Refresh the about section data after successful submission
      if (response.status === 200 || response.status === 201) {
        fetchAboutSection();
      }
    } catch (error) {
      console.error('Error:', error);
      setAboutSectionMessage({
        type: 'error',
        text: error.response?.data?.message || 'Error updating about section'
      });
    } finally {
      setAboutSectionLoading(false);
    }
  };

  // Video section handlers
  const fetchVideoSection = async () => {
    try {
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'application/json'
        }
      };
      const response = await axios.get('http://localhost:3000/api/home/video', config);
      if (response.data.success) {
        setVideoSectionData(response.data.data);
      }
    } catch (error) {
      console.error('Error fetching video section:', error);
    }
  };

  const handleVideoSectionInputChange = (e) => {
    const { name, value } = e.target;
    setVideoSectionForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleVideoChange = (e) => {
    const file = e.target.files[0];
    if (file) {
      setVideoSectionForm(prev => ({
        ...prev,
        video: file
      }));
      setVideoFileName(file.name);
    }
  };

  const handleVideoSectionSubmit = async (e) => {
    e.preventDefault();
    setVideoSectionLoading(true);
    setVideoSectionMessage({ type: '', text: '' });

    try {
      const formData = new FormData();
      formData.append('video_alt', videoSectionForm.video_alt);

      if (videoSectionForm.video) {
        formData.append('video', videoSectionForm.video);
      }

      let response;
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'multipart/form-data'
        }
      };

      // Check if video section record already exists
      if (videoSectionData && videoSectionData.length > 0) {
        // Update existing record - use PATCH
        const existingSection = videoSectionData[0];
        response = await axios.patch(`http://localhost:3000/api/home/video/${existingSection.id}`, formData, config);
        if (response.status === 200) {
          setVideoSectionMessage({ type: 'success', text: 'Video section updated successfully!' });
        }
      } else {
        // Create new record - use POST
        response = await axios.post('http://localhost:3000/api/home/video', formData, config);
        if (response.status === 201) {
          setVideoSectionMessage({ type: 'success', text: 'Video section created successfully!' });
        }
      }

      // Refresh the video section data after successful submission
      if (response.status === 200 || response.status === 201) {
        fetchVideoSection();
      }
    } catch (error) {
      console.error('Error:', error);
      setVideoSectionMessage({
        type: 'error',
        text: error.response?.data?.message || 'Error updating video section'
      });
    } finally {
      setVideoSectionLoading(false);
    }
  };

  // Achievements section handlers
  const fetchAchievementsSection = async () => {
    try {
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'application/json'
        }
      };
      const response = await axios.get('http://localhost:3000/api/home/agency-section', config);
      if (response.data.success) {
        setAchievementsSectionData(response.data.data);
      }
    } catch (error) {
      console.error('Error fetching achievements section:', error);
    }
  };

  const handleAchievementsSectionInputChange = (e) => {
    const { name, value } = e.target;
    setAchievementsSectionForm(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleAchievementsSectionSubmit = async (e) => {
    e.preventDefault();
    setAchievementsSectionLoading(true);
    setAchievementsSectionMessage({ type: '', text: '' });

    try {
      let response;
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'application/json'
        }
      };

      // Check if achievements section record already exists
      if (achievementsSectionData && achievementsSectionData.length > 0) {
        // Update existing record - use PATCH
        const existingSection = achievementsSectionData[0];
        response = await axios.patch(`http://localhost:3000/api/home/agency-section/${existingSection.id}`, achievementsSectionForm, config);
        if (response.status === 200) {
          setAchievementsSectionMessage({ type: 'success', text: 'Achievements section updated successfully!' });
        }
      } else {
        // Create new record - use POST
        response = await axios.post('http://localhost:3000/api/home/agency-section', achievementsSectionForm, config);
        if (response.status === 201) {
          setAchievementsSectionMessage({ type: 'success', text: 'Achievements section created successfully!' });
        }
      }

      // Refresh the achievements section data after successful submission
      if (response.status === 200 || response.status === 201) {
        fetchAchievementsSection();
      }
    } catch (error) {
      console.error('Error:', error);
      setAchievementsSectionMessage({
        type: 'error',
        text: error.response?.data?.message || 'Error updating achievements section'
      });
    } finally {
      setAchievementsSectionLoading(false);
    }
  };

  return (
    <Container fluid className="blogs-page">
      <Row>
        <Col>
          <div className="page-header mb-4">
            <h2 className="mb-0">Home Page Management</h2>
            <p className="text-muted" style={{ marginTop: '10px' }}>Manage your home page content and settings</p>
          </div>

          <Accordion defaultActiveKey="0" className="mb-4">
            {/* Meta Section */}
            <Accordion.Item eventKey="0">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-tags me-2"></i>
                  <span>Meta Information</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    {metaMessage.text && (
                      <Alert variant={metaMessage.type === 'success' ? 'success' : 'danger'} dismissible>
                        {metaMessage.text}
                      </Alert>
                    )}

                    <Form onSubmit={handleMetaSubmit}>
                      <Row>
                        <Col md={6}>
                          {/* Meta Title */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Meta Title *</Form.Label>
                            <Form.Control
                              type="text"
                              name="meta_title"
                              value={metaForm.meta_title}
                              onChange={handleMetaInputChange}
                              placeholder="Enter meta title"
                              required
                            />
                          </div>

                          {/* Meta Description */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Meta Description *</Form.Label>
                            <Form.Control
                              as="textarea"
                              rows={3}
                              name="meta_description"
                              value={metaForm.meta_description}
                              onChange={handleMetaInputChange}
                              placeholder="Enter meta description"
                              required
                            />
                          </div>

                          {/* Meta Keywords */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Meta Keywords</Form.Label>
                            <Form.Control
                              type="text"
                              value={metaKeywordInput}
                              onChange={handleMetaKeywordInputChange}
                              onKeyPress={handleMetaKeywordKeyPress}
                              placeholder="Type keyword and press Enter"
                            />
                            {Array.isArray(metaForm.meta_keywords) && metaForm.meta_keywords.length > 0 && (
                              <div className="keywords-tags mt-2">
                                {metaForm.meta_keywords.map((keyword, index) => (
                                  <span
                                    key={index}
                                    className="badge bg-primary me-2 mb-2 keyword-tag"
                                  >
                                    {keyword}
                                    <button
                                      type="button"
                                      className="btn-close btn-close-white ms-2"
                                      onClick={() => removeMetaKeyword(keyword)}
                                      style={{ fontSize: '8px' }}
                                    ></button>
                                  </span>
                                ))}
                              </div>
                            )}
                          </div>

                          {/* Slug */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Slug *</Form.Label>
                            <Form.Control
                              type="text"
                              name="slug"
                              value={metaForm.slug}
                              onChange={handleMetaInputChange}
                              placeholder="Enter slug"
                              required
                            />
                          </div>
                        </Col>

                        <Col md={6}>
                          {/* Canonical Tag */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Canonical Tag</Form.Label>
                            <Form.Control
                              type="url"
                              name="canonical_tag"
                              value={metaForm.canonical_tag}
                              onChange={handleMetaInputChange}
                              placeholder="https://example.com/page"
                            />
                          </div>

                          {/* OG URL */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">OG URL</Form.Label>
                            <Form.Control
                              type="url"
                              name="og_url"
                              value={metaForm.og_url}
                              onChange={handleMetaInputChange}
                              placeholder="https://example.com/page"
                            />
                          </div>

                          {/* OG Site Name */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">OG Site Name</Form.Label>
                            <Form.Control
                              type="text"
                              name="og_sitename"
                              value={metaForm.og_sitename}
                              onChange={handleMetaInputChange}
                              placeholder="Enter site name"
                            />
                          </div>

                          {/* Page For */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Page For *</Form.Label>
                            <Form.Select
                              name="page_for"
                              value={metaForm.page_for}
                              onChange={handleMetaInputChange}
                              required
                            >
                              <option value="">Select a page</option>
                              {pageOptions.map((option) => (
                                <option key={option.value} value={option.value}>
                                  {option.label}
                                </option>
                              ))}
                            </Form.Select>
                          </div>
                        </Col>
                      </Row>

                      <div className="d-flex justify-content-end">
                        <Button
                          variant="primary"
                          type="submit"
                          disabled={metaLoading}
                        >
                          {metaLoading ? 'Processing...' : (pageMeta && pageMeta.id ? 'Update Meta Information' : 'Create Meta Information')}
                        </Button>
                      </div>
                    </Form>
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* Hero Section Banners */}
            <Accordion.Item eventKey="1">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-image me-2"></i>
                  <span>Hero Section Banners ({homePageBannerJson?.length || 0})</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    <div className="d-flex justify-content-between align-items-center mb-3">
                      <h5 className="mb-0">All Hero Banners</h5>
                      <Button
                        variant="primary"
                        onClick={() => navigate('/banners')}
                      >
                        <i className="fas fa-plus me-2"></i>
                        Manage Banners
                      </Button>
                    </div>

                    {isLoading ? (
                      <div className="text-center py-5">
                        <div className="spinner-border text-primary" role="status">
                          <span className="visually-hidden">Loading...</span>
                        </div>
                        <p className="mt-2">Loading banners...</p>
                      </div>
                    ) : homePageBannerJson && homePageBannerJson.length > 0 ? (
                      <div className="banners-grid">
                        {homePageBannerJson.map((banner, index) => (
                          console.log('[banner]', banner),
                          <Card
                            key={banner.id || index}
                            className="banner-card mb-3"
                            onClick={() => handleBannerClick(banner)}
                            style={{ cursor: 'pointer' }}
                          >
                            <div className="banner-image-container">
                              <img
                                src={banner.banner_img}
                                alt={banner.img_alt || 'Banner'}
                                className="card-img-top"
                                style={{ height: '200px', objectFit: 'cover' }}
                              />
                            </div>
                            <Card.Body>
                              <div className="d-flex justify-content-between align-items-start mb-2">
                                <h6 className="card-title mb-1">{banner.main_heading}</h6>
                                <Badge bg="primary">Active</Badge>
                              </div>

                              <p className="text-muted small mb-2">
                                <strong>Content:</strong> {banner?.content?.length > 50 ? banner?.content?.slice(0, 50) + '...' : banner?.content}
                              </p>

                              <p className="text-muted small mb-2">
                                <strong>Button:</strong> {banner.button_text} → {banner.button_link}
                              </p>

                              <p className="text-muted small mb-0">
                                <strong>Created:</strong> {formatDateForUI(banner.createdAt)}
                              </p>
                            </Card.Body>

                            <Card.Footer className="bg-light">
                              <div className="d-flex justify-content-between align-items-center">
                                <small className="text-muted">
                                  Click to manage this banner
                                </small>
                                <Button variant="outline-primary" size="sm">
                                  <i className="fas fa-edit me-1"></i>
                                  Manage
                                </Button>
                              </div>
                            </Card.Footer>
                          </Card>
                        ))}
                      </div>
                    ) : (
                      <div className="text-center py-5">
                        <i className="fas fa-image fa-3x text-muted mb-3"></i>
                        <h5 className="text-muted">No Hero Banners Found</h5>
                        <p className="text-muted">Start by creating your first hero banner.</p>
                        <Button
                          variant="primary"
                          onClick={() => navigate('/banners')}
                        >
                          <i className="fas fa-plus me-2"></i>
                          Create First Banner
                        </Button>
                      </div>
                    )}
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* Our Service Section */}
            <Accordion.Item eventKey="2">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-cogs me-2"></i>
                  <span>Our Service ({servicesJson?.length || 0})</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    {/* Services Section Heading Form */}
                    <div className="mb-4">
                      <h5 className="mb-3">Section Heading</h5>
                      {servicesSectionMessage.text && (
                        <Alert variant={servicesSectionMessage.type === 'success' ? 'success' : 'danger'} dismissible>
                          {servicesSectionMessage.text}
                        </Alert>
                      )}

                      <Form onSubmit={handleServicesSectionSubmit}>
                        <Row>
                          <Col md={8}>
                            <div className="mb-3">
                              <Form.Label className="fw-bold">Section Heading *</Form.Label>
                              <Form.Control
                                type="text"
                                name="main_heading"
                                value={servicesSectionForm.main_heading}
                                onChange={handleServicesSectionInputChange}
                                placeholder="Enter section heading"
                                required
                              />
                            </div>
                          </Col>
                          <Col md={4} className="d-flex align-items-end">
                            <Button
                              variant="primary"
                              type="submit"
                              disabled={servicesSectionLoading}
                            >
                              {servicesSectionLoading ? 'Processing...' : (servicesSectionData && servicesSectionData.length > 0 ? 'Update Heading' : 'Create Heading')}
                            </Button>
                          </Col>
                        </Row>
                      </Form>
                    </div>

                    <hr className="my-4" />

                    {/* Services List */}
                    <div className="d-flex justify-content-between align-items-center mb-3">
                      <h5 className="mb-0">All Services</h5>
                      <Button
                        variant="primary"
                        onClick={() => navigate('/dashboard/services')}
                      >
                        <i className="fas fa-plus me-2"></i>
                        Manage Services
                      </Button>
                    </div>

                    {isLoading ? (
                      <div className="text-center py-5">
                        <div className="spinner-border text-primary" role="status">
                          <span className="visually-hidden">Loading...</span>
                        </div>
                        <p className="mt-2">Loading services...</p>
                      </div>
                    ) : servicesJson && servicesJson.length > 0 ? (
                      <div className="table-responsive">
                        <table className="table table-hover">
                          <thead className="table-light" style={{ backgroundColor: '#f8f9fa' }}>
                            <tr>
                              <th>Service Label</th>
                              <th>Status</th>
                              <th>Featured</th>
                              <th>Actions</th>
                            </tr>
                          </thead>
                          <tbody>
                            {servicesJson.map((service, index) => (
                              <tr
                                key={service.id || index}
                                onClick={() => handleServiceClick(service)}
                                style={{ cursor: 'pointer' }}
                              >
                                <td>
                                  <span>{service.service_label}</span>
                                </td>
                                <td>
                                  <Badge bg={service.is_active ? 'success' : 'secondary'}>
                                    {service.is_active ? 'Active' : 'Inactive'}
                                  </Badge>
                                </td>
                                <td>
                                  <Badge bg={service.is_featured ? 'warning' : 'info'} text={service.is_featured ? 'dark' : 'dark'}>
                                    {service.is_featured ? 'Featured' : 'Not Featured'}
                                  </Badge>
                                </td>
                                <td>
                                  <Button variant="outline-primary" size="sm">
                                    <i className="fas fa-edit me-1"></i>
                                    Manage
                                  </Button>
                                </td>
                              </tr>
                            ))}
                          </tbody>
                        </table>
                      </div>
                    ) : (
                      <div className="text-center py-5">
                        <i className="fas fa-cogs fa-3x text-muted mb-3"></i>
                        <h5 className="text-muted">No Services Found</h5>
                        <p className="text-muted">Start by creating your first service.</p>
                        <Button
                          variant="primary"
                          onClick={() => navigate('/dashboard/services')}
                        >
                          <i className="fas fa-plus me-2"></i>
                          Create First Service
                        </Button>
                      </div>
                    )}
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* About Section */}
            <Accordion.Item eventKey="3">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-info-circle me-2"></i>
                  <span>About Section</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    {aboutSectionMessage.text && (
                      <Alert variant={aboutSectionMessage.type === 'success' ? 'success' : 'danger'} dismissible>
                        {aboutSectionMessage.text}
                      </Alert>
                    )}

                    <Form onSubmit={handleAboutSectionSubmit}>
                      <Row>
                        <Col md={8}>
                          {/* Span Heading */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Span Heading *</Form.Label>
                            <Form.Control
                              type="text"
                              name="span_heading"
                              value={aboutSectionForm.span_heading}
                              onChange={handleAboutSectionInputChange}
                              placeholder="e.g., WHO WE ARE"
                              required
                            />
                          </div>

                          {/* Main Heading 1 */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Main Heading 1 *</Form.Label>
                            <Form.Control
                              type="text"
                              name="main_heading_1"
                              value={aboutSectionForm.main_heading_1}
                              onChange={handleAboutSectionInputChange}
                              placeholder="e.g., Engineering"
                              required
                            />
                          </div>

                          {/* Main Heading 2 */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Main Heading 2 *</Form.Label>
                            <Form.Control
                              type="text"
                              name="main_heading_2"
                              value={aboutSectionForm.main_heading_2}
                              onChange={handleAboutSectionInputChange}
                              placeholder="e.g., With Purpose"
                              required
                            />
                          </div>

                          {/* Content Editor */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Content *</Form.Label>
                            <div className="quill-editor-container">
                              <ReactQuill
                                theme="snow"
                                value={aboutSectionForm.content}
                                onChange={handleAboutContentChange}
                                modules={quillModules}
                                formats={quillFormats}
                                placeholder="Enter about section content..."
                                style={{ height: '200px' }}
                              />
                            </div>
                          </div>

                          {/* Button Text */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Button Text *</Form.Label>
                            <Form.Control
                              type="text"
                              name="button_text"
                              value={aboutSectionForm.button_text}
                              onChange={handleAboutSectionInputChange}
                              placeholder="e.g., Learn More"
                              required
                            />
                          </div>

                          {/* Button Link */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Button Link *</Form.Label>
                            <Form.Control
                              type="url"
                              name="button_link"
                              value={aboutSectionForm.button_link}
                              onChange={handleAboutSectionInputChange}
                              placeholder="https://example.com/about"
                              required
                            />
                          </div>
                        </Col>

                        <Col md={4}>
                          {/* Main Heading Image */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Main Heading Image</Form.Label>
                            <div className="image-upload-section">
                              <Form.Control
                                type="file"
                                accept="image/*"
                                onChange={handleAboutImageChange}
                                className="mb-2"
                              />
                              {aboutImagePreview && (
                                <div className="image-preview">
                                  <img
                                    src={aboutImagePreview}
                                    alt="About Section Preview"
                                    className="img-thumbnail"
                                    style={{ maxWidth: '100%', height: '150px', objectFit: 'cover' }}
                                  />
                                </div>
                              )}
                            </div>
                          </div>
                        </Col>
                      </Row>

                      <div className="d-flex justify-content-end">
                        <Button
                          variant="primary"
                          type="submit"
                          disabled={aboutSectionLoading}
                        >
                          {aboutSectionLoading ? 'Processing...' : (aboutSectionData && aboutSectionData.length > 0 ? 'Update About Section' : 'Create About Section')}
                        </Button>
                      </div>
                    </Form>
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* Video Section */}
            <Accordion.Item eventKey="4">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-video me-2"></i>
                  <span>Video Section</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    {videoSectionMessage.text && (
                      <Alert variant={videoSectionMessage.type === 'success' ? 'success' : 'danger'} dismissible>
                        {videoSectionMessage.text}
                      </Alert>
                    )}

                    <Form onSubmit={handleVideoSectionSubmit}>
                      <Row>
                        <Col md={8}>
                          {/* Video Alt Text */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Video Alt Text *</Form.Label>
                            <Form.Control
                              type="text"
                              name="video_alt"
                              value={videoSectionForm.video_alt}
                              onChange={handleVideoSectionInputChange}
                              placeholder="Enter video alt text"
                              required
                            />
                          </div>
                        </Col>

                        <Col md={4}>
                          {/* Video Upload */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Video File</Form.Label>
                            <div className="video-upload-section">
                              <Form.Control
                                type="file"
                                accept="video/*"
                                onChange={handleVideoChange}
                                className="mb-2"
                              />
                              <small className="text-muted">
                                {videoFileName}
                              </small>
                              {!videoSectionData && (
                                <small className="text-danger d-block">* Video file is required for new uploads</small>
                              )}
                            </div>
                          </div>
                        </Col>
                      </Row>

                      <div className="d-flex justify-content-end">
                        <Button
                          variant="primary"
                          type="submit"
                          disabled={videoSectionLoading}
                        >
                          {videoSectionLoading ? 'Processing...' : (videoSectionData && videoSectionData.length > 0 ? 'Update Video' : 'Upload Video')}
                        </Button>
                      </div>
                    </Form>
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* Achievements Section */}
            <Accordion.Item eventKey="5">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-trophy me-2"></i>
                  <span>Achievements Section</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    {achievementsSectionMessage.text && (
                      <Alert variant={achievementsSectionMessage.type === 'success' ? 'success' : 'danger'} dismissible>
                        {achievementsSectionMessage.text}
                      </Alert>
                    )}

                    <Form onSubmit={handleAchievementsSectionSubmit}>
                      <Row>
                        <Col md={6}>
                          {/* Main Heading 1 */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Main Heading 1 *</Form.Label>
                            <Form.Control
                              type="text"
                              name="main_heading_1"
                              value={achievementsSectionForm.main_heading_1}
                              onChange={handleAchievementsSectionInputChange}
                              placeholder="e.g., main_heading_1update"
                              required
                            />
                          </div>

                          {/* Main Heading 2 */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Main Heading 2 *</Form.Label>
                            <Form.Control
                              type="text"
                              name="main_heading_2"
                              value={achievementsSectionForm.main_heading_2}
                              onChange={handleAchievementsSectionInputChange}
                              placeholder="e.g., We deliver purpose update"
                              required
                            />
                          </div>
                        </Col>

                        <Col md={6}>
                          {/* Content Paragraph 1 */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Content Paragraph 1 *</Form.Label>
                            <Form.Control
                              as="textarea"
                              rows={3}
                              name="content_para_1"
                              value={achievementsSectionForm.content_para_1}
                              onChange={handleAchievementsSectionInputChange}
                              placeholder="e.g., We empower digital transformation update"
                              required
                            />
                          </div>

                          {/* Content Paragraph 2 */}
                          <div className="mb-4">
                            <Form.Label className="fw-bold">Content Paragraph 2 *</Form.Label>
                            <Form.Control
                              as="textarea"
                              rows={3}
                              name="content_para_2"
                              value={achievementsSectionForm.content_para_2}
                              onChange={handleAchievementsSectionInputChange}
                              placeholder="e.g., Empowering design with technology update"
                              required
                            />
                          </div>
                        </Col>
                      </Row>

                      <div className="d-flex justify-content-end">
                        <Button
                          variant="primary"
                          type="submit"
                          disabled={achievementsSectionLoading}
                        >
                          {achievementsSectionLoading ? 'Processing...' : (achievementsSectionData && achievementsSectionData.length > 0 ? 'Update Achievements Section' : 'Create Achievements Section')}
                        </Button>
                      </div>
                    </Form>

                    {/* Achievements List */}
                    <div className="mt-5">
                      <div className="d-flex justify-content-between align-items-center mb-3">
                        <h5 className="mb-0">All Achievements</h5>
                        <Button
                          variant="primary"
                          onClick={() => navigate('/dashboard/achievements')}
                        >
                          <i className="fas fa-plus me-2"></i>
                          Manage Achievements
                        </Button>
                      </div>

                      {isLoading ? (
                        <div className="text-center py-5">
                          <div className="spinner-border text-primary" role="status">
                            <span className="visually-hidden">Loading...</span>
                          </div>
                          <p className="mt-2">Loading achievements...</p>
                        </div>
                      ) : achievementsJson && achievementsJson.length > 0 ? (
                        <div className="table-responsive">
                          <table className="table table-hover">
                            <thead style={{ backgroundColor: '#f8f9fa' }}>
                              <tr>
                                <th>Image</th>
                                <th>Card Title</th>
                                <th>Actions</th>
                              </tr>
                            </thead>
                            <tbody>
                              {achievementsJson.map((achievement, index) => (
                                <tr
                                  key={achievement.id || index}
                                  onClick={() => handleAchievementClick(achievement)}
                                  style={{ cursor: 'pointer' }}
                                  className="achievement-row"
                                >
                                  <td>
                                    {achievement.image_path ? (
                                      <img
                                        src={achievement.image_path}
                                        alt={achievement.card_title || 'Achievement'}
                                        className="img-thumbnail"
                                        style={{ width: '50px', height: '50px', objectFit: 'cover' }}
                                      />
                                    ) : (
                                      <div
                                        className="bg-secondary d-flex align-items-center justify-content-center"
                                        style={{ width: '50px', height: '50px' }}
                                      >
                                        <i className="fas fa-image text-white"></i>
                                      </div>
                                    )}
                                  </td>
                                  <td>
                                    <span className="fw-medium">{achievement.card_title || 'N/A'}</span>
                                  </td>
                                  <td>
                                    <Button variant="outline-primary" size="sm">
                                      <i className="fas fa-edit me-1"></i>
                                      Edit
                                    </Button>
                                  </td>
                                </tr>
                              ))}
                            </tbody>
                          </table>
                        </div>
                      ) : (
                        <div className="text-center py-5">
                          <i className="fas fa-trophy fa-3x text-muted mb-3"></i>
                          <h5 className="text-muted">No Achievements Found</h5>
                          <p className="text-muted">Start by creating your first achievement.</p>
                          <Button
                            variant="primary"
                            onClick={() => navigate('/dashboard/achievements')}
                          >
                            <i className="fas fa-plus me-2"></i>
                            Create First Achievement
                          </Button>
                        </div>
                      )}
                    </div>
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* Promotion Section */}
            <Accordion.Item eventKey="6">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-bullhorn me-2"></i>
                  <span>Promotion Section</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    <div className="d-flex justify-content-between align-items-center mb-3">
                      <h5 className="mb-0">All Promotions</h5>
                      <Button
                        variant="primary"
                        onClick={() => navigate('/dashboard/promotions')}
                      >
                        <i className="fas fa-plus me-2"></i>
                        Manage Promotions
                      </Button>
                    </div>

                    {isLoading ? (
                      <div className="text-center py-5">
                        <div className="spinner-border text-primary" role="status">
                          <span className="visually-hidden">Loading...</span>
                        </div>
                        <p className="mt-2">Loading promotions...</p>
                      </div>
                    ) : promotionsJson && promotionsJson.length > 0 ? (
                      <div className="table-responsive">
                        <table className="table table-hover">
                          <thead style={{ backgroundColor: '#f8f9fa' }}>
                            <tr>
                              <th>Image</th>
                              <th>Image Alt</th>
                              <th>Logo For</th>
                              <th>Actions</th>
                            </tr>
                          </thead>
                          <tbody>
                            {promotionsJson.map((promotion, index) => (
                              <tr
                                key={promotion.id || index}
                                onClick={() => handlePromotionClick(promotion)}
                                style={{ cursor: 'pointer' }}
                                className="promotion-row"
                              >
                                <td>
                                  {promotion.image_path ? (
                                    <img
                                      src={promotion.image_path}
                                      alt={promotion.img_alt || 'Promotion'}
                                      className="img-thumbnail"
                                      style={{ width: '50px', height: '50px', objectFit: 'cover' }}
                                    />
                                  ) : (
                                    <div
                                      className="bg-secondary d-flex align-items-center justify-content-center"
                                      style={{ width: '50px', height: '50px' }}
                                    >
                                      <i className="fas fa-image text-white"></i>
                                    </div>
                                  )}
                                </td>
                                <td>
                                  <span className="fw-medium">{promotion.img_alt || 'N/A'}</span>
                                </td>
                                <td>
                                  {promotion.logo_for === 'Award_logo' ? (
                                    <Badge bg="success" className="py-2 px-3">Award</Badge>
                                  ) : promotion.logo_for === 'Partners_logo' ? (
                                    <Badge bg="primary" className="py-2 px-3">Partners</Badge>
                                  ) : (
                                    <Badge bg="secondary" className="py-2 px-3">{promotion.logo_for || 'N/A'}</Badge>
                                  )}
                                </td>
                                <td>
                                  <Button variant="outline-primary" size="sm">
                                    <i className="fas fa-edit me-1"></i>
                                    Edit
                                  </Button>
                                </td>
                              </tr>
                            ))}
                          </tbody>
                        </table>
                      </div>
                    ) : (
                      <div className="text-center py-5">
                        <i className="fas fa-bullhorn fa-3x text-muted mb-3"></i>
                        <h5 className="text-muted">No Promotions Found</h5>
                        <p className="text-muted">Start by creating your first promotion.</p>
                        <Button
                          variant="primary"
                          onClick={() => navigate('/dashboard/promotions')}
                        >
                          <i className="fas fa-plus me-2"></i>
                          Create First Promotion
                        </Button>
                      </div>
                    )}
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* Services Provide Section */}
            <Accordion.Item eventKey="7">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-cogs me-2"></i>
                  <span>Services Provide</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    <div className="d-flex justify-content-between align-items-center mb-3">
                      <h5 className="mb-0">All Services We Provide</h5>
                      <Button
                        variant="primary"
                        onClick={() => navigate('/dashboard/services-tabs')}
                      >
                        <i className="fas fa-plus me-2"></i>
                        Manage Services
                      </Button>
                    </div>

                    {isLoading ? (
                      <div className="text-center py-5">
                        <div className="spinner-border text-primary" role="status">
                          <span className="visually-hidden">Loading...</span>
                        </div>
                        <p className="mt-2">Loading services...</p>
                      </div>
                    ) : servicesTabsJson && servicesTabsJson.length > 0 ? (
                      <div className="table-responsive">
                        <table className="table table-hover">
                          <thead style={{ backgroundColor: '#f8f9fa' }}>
                            <tr>
                              <th>Service Label</th>
                              <th>Actions</th>
                            </tr>
                          </thead>
                          <tbody>
                            {servicesTabsJson.map((serviceTab, index) => (
                              <tr
                                key={serviceTab.id || index}
                                onClick={() => handleServiceTabClick(serviceTab)}
                                style={{ cursor: 'pointer' }}
                                className="service-tab-row"
                              >
                                <td>
                                  <span className="fw-medium">{serviceTab.service_label || 'N/A'}</span>
                                </td>
                                <td>
                                  <Button variant="outline-primary" size="sm">
                                    <i className="fas fa-edit me-1"></i>
                                    Edit
                                  </Button>
                                </td>
                              </tr>
                            ))}
                          </tbody>
                        </table>
                      </div>
                    ) : (
                      <div className="text-center py-5">
                        <i className="fas fa-cogs fa-3x text-muted mb-3"></i>
                        <h5 className="text-muted">No Services Found</h5>
                        <p className="text-muted">Start by creating your first service.</p>
                        <Button
                          variant="primary"
                          onClick={() => navigate('/dashboard/services-tabs')}
                        >
                          <i className="fas fa-plus me-2"></i>
                          Create First Service
                        </Button>
                      </div>
                    )}
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* Portfolio Section */}
            <Accordion.Item eventKey="8">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-briefcase me-2"></i>
                  <span>Portfolio</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    <div className="d-flex justify-content-between align-items-center mb-3">
                      <h5 className="mb-0">All Portfolio Items</h5>
                      <Button
                        variant="primary"
                        onClick={() => navigate('/dashboard/portfolio')}
                      >
                        <i className="fas fa-plus me-2"></i>
                        Manage Portfolio
                      </Button>
                    </div>

                    {isLoading ? (
                      <div className="text-center py-5">
                        <div className="spinner-border text-primary" role="status">
                          <span className="visually-hidden">Loading...</span>
                        </div>
                        <p className="mt-2">Loading portfolio items...</p>
                      </div>
                    ) : portfolioListingJson && portfolioListingJson.length > 0 ? (
                      <div className="table-responsive">
                        <table className="table table-hover">
                          <thead style={{ backgroundColor: '#f8f9fa' }}>
                            <tr>
                              <th>Image</th>
                              <th>Title</th>
                              <th>Link</th>
                              <th>Actions</th>
                            </tr>
                          </thead>
                          <tbody>
                            {portfolioListingJson.map((portfolio, index) => (
                              <tr 
                                key={portfolio.id || index}
                                onClick={() => handlePortfolioClick(portfolio)}
                                style={{ cursor: 'pointer' }}
                                className="portfolio-row"
                              >
                                <td>
                                  {portfolio.media_url ? (
                                    <img
                                      src={portfolio.media_url}
                                      alt={portfolio.item_heading || 'Portfolio'}
                                      className="img-thumbnail"
                                      style={{ width: '50px', height: '50px', objectFit: 'cover' }}
                                    />
                                  ) : (
                                    <div 
                                      className="bg-secondary d-flex align-items-center justify-content-center"
                                      style={{ width: '50px', height: '50px' }}
                                    >
                                      <i className="fas fa-image text-white"></i>
                                    </div>
                                  )}
                                </td>
                                <td>
                                  <span className="fw-medium">{portfolio.item_heading || 'N/A'}</span>
                                </td>
                                <td>
                                  {portfolio.item_link ? (
                                    <a 
                                      href={portfolio.item_link} 
                                      target="_blank" 
                                      rel="noopener noreferrer"
                                      className="text-primary text-decoration-none"
                                      onClick={(e) => e.stopPropagation()}
                                    >
                                      <i className="fas fa-external-link-alt me-1"></i>
                                      View Link
                                    </a>
                                  ) : (
                                    <span className="text-muted">No link</span>
                                  )}
                                </td>
                                <td>
                                  <Button variant="outline-primary" size="sm">
                                    <i className="fas fa-edit me-1"></i>
                                    Edit
                                  </Button>
                                </td>
                              </tr>
                            ))}
                          </tbody>
                        </table>
                      </div>
                    ) : (
                      <div className="text-center py-5">
                        <i className="fas fa-briefcase fa-3x text-muted mb-3"></i>
                        <h5 className="text-muted">No Portfolio Items Found</h5>
                        <p className="text-muted">Start by creating your first portfolio item.</p>
                        <Button
                          variant="primary"
                          onClick={() => navigate('/dashboard/portfolio')}
                        >
                          <i className="fas fa-plus me-2"></i>
                          Create First Portfolio Item
                        </Button>
                      </div>
                    )}
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* Reviews Section */}
            <Accordion.Item eventKey="9">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-star me-2"></i>
                  <span>Reviews</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    <div className="d-flex justify-content-between align-items-center mb-3">
                      <h5 className="mb-0">All Reviews</h5>
                      <Button
                        variant="primary"
                        onClick={() => navigate('/dashboard/reviews')}
                      >
                        <i className="fas fa-plus me-2"></i>
                        Manage Reviews
                      </Button>
                    </div>

                    {isLoading ? (
                      <div className="text-center py-5">
                        <div className="spinner-border text-primary" role="status">
                          <span className="visually-hidden">Loading...</span>
                        </div>
                        <p className="mt-2">Loading reviews...</p>
                      </div>
                    ) : reviewsJson && reviewsJson.length > 0 ? (
                      <div className="table-responsive">
                        <table className="table table-hover">
                          <thead style={{ backgroundColor: '#f8f9fa' }}>
                            <tr>
                              <th>Thumbnail</th>
                              <th>Reviewer Title</th>
                              <th>Service Name</th>
                              <th>Actions</th>
                            </tr>
                          </thead>
                          <tbody>
                            {reviewsJson.map((review, index) => (
                              <tr 
                                key={review.id || index}
                                onClick={() => handleReviewClick(review)}
                                style={{ cursor: 'pointer' }}
                                className="review-row"
                              >
                                <td>
                                  {review.reviewer_thumbnail ? (
                                    <img
                                      src={review.reviewer_thumbnail}
                                      alt={review.reviewer_title || 'Reviewer'}
                                      className="img-thumbnail"
                                      style={{ width: '50px', height: '50px', objectFit: 'cover' }}
                                    />
                                  ) : (
                                    <div 
                                      className="bg-secondary d-flex align-items-center justify-content-center"
                                      style={{ width: '50px', height: '50px' }}
                                    >
                                      <i className="fas fa-user text-white"></i>
                                    </div>
                                  )}
                                </td>
                                <td>
                                  <span className="fw-medium">{review.reviewer_title || 'N/A'}</span>
                                </td>
                                <td>
                                  <span className="text-muted">{review.reviewer_service_name || 'N/A'}</span>
                                </td>
                                <td>
                                  <Button variant="outline-primary" size="sm">
                                    <i className="fas fa-edit me-1"></i>
                                    Edit
                                  </Button>
                                </td>
                              </tr>
                            ))}
                          </tbody>
                        </table>
                      </div>
                    ) : (
                      <div className="text-center py-5">
                        <i className="fas fa-star fa-3x text-muted mb-3"></i>
                        <h5 className="text-muted">No Reviews Found</h5>
                        <p className="text-muted">Start by creating your first review.</p>
                        <Button
                          variant="primary"
                          onClick={() => navigate('/dashboard/reviews')}
                        >
                          <i className="fas fa-plus me-2"></i>
                          Create First Review
                        </Button>
                      </div>
                    )}
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>

            {/* FAQs Section */}
            <Accordion.Item eventKey="10">
              <Accordion.Header>
                <div className="d-flex align-items-center">
                  <i className="fas fa-question-circle me-2"></i>
                  <span>FAQs</span>
                </div>
              </Accordion.Header>
              <Accordion.Body>
                <Card className="border-0 shadow-sm">
                  <Card.Body>
                    <div className="d-flex justify-content-between align-items-center mb-3">
                      <h5 className="mb-0">All FAQs</h5>
                      <Button
                        variant="primary"
                        onClick={() => navigate('/dashboard/faqs')}
                      >
                        <i className="fas fa-plus me-2"></i>
                        Manage FAQs
                      </Button>
                    </div>

                    {isLoading ? (
                      <div className="text-center py-5">
                        <div className="spinner-border text-primary" role="status">
                          <span className="visually-hidden">Loading...</span>
                        </div>
                        <p className="mt-2">Loading FAQs...</p>
                      </div>
                    ) : faqsJson && faqsJson.length > 0 ? (
                      <div className="table-responsive">
                        <table className="table table-hover">
                          <thead style={{ backgroundColor: '#f8f9fa' }}>
                            <tr>
                              <th>Question</th>
                              <th>Answer</th>
                              <th>Actions</th>
                            </tr>
                          </thead>
                          <tbody>
                            {faqsJson.map((faq, index) => (
                              <tr 
                                key={faq.id || index}
                                onClick={() => handleFaqClick(faq)}
                                style={{ cursor: 'pointer' }}
                                className="faq-row"
                              >
                                <td>
                                  <span className="fw-medium">{faq.question || 'N/A'}</span>
                                </td>
                                <td>
                                  <div 
                                    className="text-muted"
                                    style={{ 
                                      maxWidth: '400px', 
                                      overflow: 'hidden', 
                                      textOverflow: 'ellipsis', 
                                      whiteSpace: 'nowrap' 
                                    }}
                                    title={faq.answer}
                                  >
                                    {faq.answer || 'N/A'}
                                  </div>
                                </td>
                                <td>
                                  <Button variant="outline-primary" size="sm">
                                    <i className="fas fa-edit me-1"></i>
                                    Edit
                                  </Button>
                                </td>
                              </tr>
                            ))}
                          </tbody>
                        </table>
                      </div>
                    ) : (
                      <div className="text-center py-5">
                        <i className="fas fa-question-circle fa-3x text-muted mb-3"></i>
                        <h5 className="text-muted">No FAQs Found</h5>
                        <p className="text-muted">Start by creating your first FAQ.</p>
                        <Button
                          variant="primary"
                          onClick={() => navigate('/dashboard/faqs')}
                        >
                          <i className="fas fa-plus me-2"></i>
                          Create First FAQ
                        </Button>
                      </div>
                    )}
                  </Card.Body>
                </Card>
              </Accordion.Body>
            </Accordion.Item>
          </Accordion>
        </Col>
      </Row>
    </Container>
  );
};

export default HomePage;