import { Col, Row } from 'react-bootstrap'
import { ButtonComp, InputComp } from '../../bootstrap'
import { useModal } from '../../../hooks/useModal'
import { useEffect, useState, useRef } from 'react'
import { useDispatch } from 'react-redux'
import { errorNotify, successNotify } from '../../../Toastify/Toastify'
import axios from 'axios'
import { BASE_URL } from '../../../config/service'
import { GetAllServices, GetHomePageBanners } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice'
import ReactQuill from 'react-quill'
import 'react-quill/dist/quill.snow.css'

const AddServiceType = ({ isEdit, data, setIsEdit, onSuccess }) => {
    console.log('[data]',data)
    const { handleModalClose } = useModal()
    const dispatch = useDispatch();
    const fileInputRef = useRef();
    
    // Service form states
    const [serviceLabel, setServiceLabel] = useState("")
    const [contentHeading, setContentHeading] = useState("")
    const [serviceContent, setServiceContent] = useState("")
    const [imgAlt, setImgAlt] = useState("")
    const [serviceImg, setServiceImg] = useState(null)
    const [fileName, setFileName] = useState('No file chosen')
    const [isSubmitting, setIsSubmitting] = useState(false)
    const [isActive, setIsActive] = useState(true)
    const [showOnHome, setShowOnHome] = useState(true)
    const [isFeatured, setIsFeatured] = useState(true)

    // Quill editor configuration
    const quillModules = {
        toolbar: [
            [{ 'header': [1, 2, 3, 4, 5, 6, false] }],
            ['bold', 'italic', 'underline', 'strike'],
            [{ 'list': 'ordered'}, { 'list': 'bullet' }],
            [{ 'color': [] }, { 'background': [] }],
            [{ 'align': [] }],
            ['link', 'image'],
            ['clean']
        ],
    };

    const quillFormats = [
        'header',
        'bold', 'italic', 'underline', 'strike',
        'list', 'bullet',
        'color', 'background',
        'align',
        'link', 'image'
    ];

    useEffect(() => {
        if (isEdit && data) {
            setServiceLabel(data.service_label || "")
            setContentHeading(data.content_heading || "")
            setServiceContent(data.service_content || "")
            setImgAlt(data.img_alt || "")
            setFileName(data.service_img ? 'Current image' : 'No file chosen')
            setIsActive(data.is_active !== undefined ? data.is_active : true)
            setShowOnHome(data.show_on_home !== undefined ? data.show_on_home : true)
            setIsFeatured(data.is_featured !== undefined ? data.is_featured : true)
        }
    }, [isEdit, data])

    const handleFileChange = (event) => {
        const file = event.target.files[0];
        if (file) {
            setServiceImg(file);
            setFileName(file.name);
        } else {
            setServiceImg(null);
            setFileName('No file chosen');
        }
    };

    const handleButtonClick = () => {
        fileInputRef.current.click();
    };

    const addServiceHandler = async (e) => {
        e.preventDefault()
        setIsSubmitting(true)

        // Validation
        if (!serviceLabel || !contentHeading || !serviceContent || !imgAlt) {
            errorNotify("Required Fields are missing!")
            setIsSubmitting(false)
            return
        }

        try {
            const formData = new FormData();
            formData.append('service_label', serviceLabel);
            formData.append('content_heading', contentHeading);
            formData.append('service_content', serviceContent);
            formData.append('img_alt', imgAlt);
            formData.append('is_active', isActive.toString());
            formData.append('show_on_home', showOnHome.toString());
            formData.append('is_featured', isFeatured.toString());
            
            if (serviceImg) {
                formData.append('service_img', serviceImg);
            }

            // Get auth token from localStorage
            const token = localStorage.getItem('token');
            if (!token) {
                errorNotify("Authentication token not found!")
                setIsSubmitting(false)
                return
            }

            const config = {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'multipart/form-data'
                }
            };

            let response;
            if (!isEdit) {
                // Create new service
                response = await axios.post(`${BASE_URL}/services`, formData, config);
                successNotify("Service added successfully!");
            } else {
                // Update existing service
                response = await axios.patch(`${BASE_URL}/services/${data.id}`, formData, config);
                successNotify("Service updated successfully!");
            }
            dispatch(GetAllServices())

            handleModalClose();
            setIsEdit(false);
            
            // Reset form
            setServiceLabel("");
            setContentHeading("");
            setServiceContent("");
            setImgAlt("");
            setServiceImg(null);
            setFileName('No file chosen');
            setIsActive(true);
            setShowOnHome(true);
            setIsFeatured(true);
            
            // Call success callback if provided
            if (onSuccess) {
                onSuccess();
            }

        } catch (error) {
            console.error('Service submission error:', error);
            const errorMessage = error.response?.data?.message || error.message || "Something went wrong!";
            errorNotify(errorMessage);
        } finally {
            setIsSubmitting(false);
        }
    }

    return (
        <fieldset className="reset mt-3 p-4 pt-0 t-0 ">
            <legend className="reset mb-3 side-heading px-5 mx-auto">{isEdit ? 'Update' : 'Add'} Service</legend>
            <form onSubmit={addServiceHandler} >
                <Row className='mt-3'>
                    <Col xxl={12} className="">
                        <Row>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Service Label</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setServiceLabel(e.target.value)} 
                                        required={true} 
                                        label={"Enter Service Label: *"} 
                                        type={"text"} 
                                        placeholder={"Service Label"} 
                                        controlId={"floatingInput-1"} 
                                        name="serviceLabel" 
                                        value={serviceLabel} 
                                    />
                                </div>
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Content Heading</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setContentHeading(e.target.value)} 
                                        required={true} 
                                        label={"Enter Content Heading: *"} 
                                        type={"text"} 
                                        placeholder={"Content Heading"} 
                                        controlId={"floatingInput-2"} 
                                        name="contentHeading" 
                                        value={contentHeading} 
                                    />
                                </div>
                            </Col>
                        </Row>
                        
                        <Row className='mt-3'>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Service Content</h5>
                                <div className='property-no'>
                                    <style>
                                        {`
                                            .ql-editor {
                                                min-height: 150px;
                                                font-size: 14px;
                                                line-height: 1.6;
                                            }
                                            .ql-toolbar {
                                                border-top: 1px solid #ccc;
                                                border-left: 1px solid #ccc;
                                                border-right: 1px solid #ccc;
                                                border-radius: 4px 4px 0 0;
                                            }
                                            .ql-container {
                                                border-bottom: 1px solid #ccc;
                                                border-left: 1px solid #ccc;
                                                border-right: 1px solid #ccc;
                                                border-radius: 0 0 4px 4px;
                                            }
                                            .ql-editor.ql-blank::before {
                                                color: #6c757d;
                                                font-style: italic;
                                            }
                                        `}
                                    </style>
                                    <ReactQuill
                                        theme="snow"
                                        value={serviceContent}
                                        onChange={setServiceContent}
                                        modules={quillModules}
                                        formats={quillFormats}
                                        placeholder="Enter service content..."
                                        style={{ 
                                            marginBottom: '50px'
                                        }}
                                    />
                                </div>
                            </Col>
                        </Row>

                        <Row className='mt-3'>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>Service Settings</h5>
                                <div className="row">
                                    <div className="col-md-4">
                                        <div className="form-check">
                                            <input
                                                className="form-check-input"
                                                type="checkbox"
                                                id="isActiveCheckbox"
                                                checked={isActive}
                                                onChange={(e) => setIsActive(e.target.checked)}
                                            />
                                            <label className="form-check-label" htmlFor="isActiveCheckbox">
                                                <strong>Active Service</strong>
                                            </label>
                                        </div>
                                    </div>
                                    <div className="col-md-4">
                                        <div className="form-check">
                                            <input
                                                className="form-check-input"
                                                type="checkbox"
                                                id="showOnHomeCheckbox"
                                                checked={showOnHome}
                                                onChange={(e) => setShowOnHome(e.target.checked)}
                                            />
                                            <label className="form-check-label" htmlFor="showOnHomeCheckbox">
                                                <strong>Show on Homepage</strong>
                                            </label>
                                        </div>
                                    </div>
                                    <div className="col-md-4">
                                        <div className="form-check">
                                            <input
                                                className="form-check-input"
                                                type="checkbox"
                                                id="isFeaturedCheckbox"
                                                checked={isFeatured}
                                                onChange={(e) => setIsFeatured(e.target.checked)}
                                            />
                                            <label className="form-check-label" htmlFor="isFeaturedCheckbox">
                                                <strong>Featured Service</strong>
                                            </label>
                                        </div>
                                    </div>
                                </div>
                            </Col>
                        </Row>

                        <Row className='mt-3'>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Service Image</h5>
                                <div className='file-upload_comp'>
                                    <input
                                        type="file"
                                        ref={fileInputRef}
                                        onChange={handleFileChange}
                                        accept="image/*"
                                        style={{ display: 'none' }}
                                    />
                                    <button 
                                        type="button" 
                                        onClick={handleButtonClick}
                                        className="btn btn-outline-primary"
                                    >
                                        Choose file
                                    </button>
                                    <span className="ms-2">{fileName}</span>
                                </div>
                                {!isEdit && (
                                    <small className="text-danger">* Service image is required for new services</small>
                                )}
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Image Alt Text</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setImgAlt(e.target.value)} 
                                        required={true} 
                                        label={"Enter Image Alt Text: *"} 
                                        type={"text"} 
                                        placeholder={"Image Alt Text"} 
                                        controlId={"floatingInput-5"} 
                                        name="imgAlt" 
                                        value={imgAlt} 
                                    />
                                </div>
                            </Col>
                        </Row>

                        <Row>
                            <Col lg={12} className=' mt-5'>
                                <ButtonComp 
                                    type="submit" 
                                    variant={"outline-primary"} 
                                    className="btn-main-clr text-white" 
                                    btnText={isSubmitting ? 'Processing...' : (isEdit ? 'Update Service' : 'Add Service')} 
                                    disabled={isSubmitting}
                                />
                            </Col>
                        </Row>
                    </Col>
                </Row>
            </form>
        </fieldset>
    )
}

export default AddServiceType