import { Col, Row } from 'react-bootstrap'
import { ButtonComp, InputComp } from '../../bootstrap'
import { useModal } from '../../../hooks/useModal'
import { useEffect, useState } from 'react'
import { useDispatch } from 'react-redux'
import { errorNotify, successNotify } from '../../../Toastify/Toastify'
import axios from 'axios'
import { BASE_URL } from '../../../config/service'
import { GetAllServicesTabs } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice'

const AddServiceTab = ({ isEdit, data, setIsEdit, onSuccess }) => {
    console.log('[data]',data)
    const { handleModalClose } = useModal()
    const dispatch = useDispatch();
    
    // Service Tab form states
    const [label, setLabel] = useState("")
    const [content, setContent] = useState([])
    const [contentInput, setContentInput] = useState('')
    const [isSubmitting, setIsSubmitting] = useState(false)



    useEffect(() => {
        if (isEdit && data) {
            setLabel(data.service_label || "")
            // Parse content from JSON string to array if needed
            let contentArray = [];
            try {
                contentArray = data.service_content ? (Array.isArray(data.service_content) ? data.service_content : JSON.parse(data.service_content)) : [];
            } catch (error) {
                console.error('Error parsing content:', error);
                contentArray = [];
            }
            setContent(contentArray)
        }
    }, [isEdit, data])



    const handleContentInputChange = (e) => {
        setContentInput(e.target.value);
    };

    const handleContentKeyPress = (e) => {
        if (e.key === 'Enter' && contentInput.trim()) {
            e.preventDefault();
            const contentItem = contentInput.trim();

            if (!content.includes(contentItem)) {
                setContent(prev => [...prev, contentItem]);
            }
            setContentInput('');
        }
    };

    const removeContent = (contentToRemove) => {
        setContent(prev => prev.filter(item => item !== contentToRemove));
    };

    const addServiceTabHandler = async (e) => {
        e.preventDefault()
        setIsSubmitting(true)

        // Validation
        if (!label || content.length === 0) {
            errorNotify("Label and at least one content item are required!")
            setIsSubmitting(false)
            return
        }

        try {
            const payload = {
                label: label,
                content: content
            };

            // Get auth token from localStorage
            const token = localStorage.getItem('token');
            if (!token) {
                errorNotify("Authentication token not found!")
                setIsSubmitting(false)
                return
            }

            const config = {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json'
                }
            };

            let response;
            if (!isEdit) {
                // Create new service tab
                response = await axios.post(`${BASE_URL}/services-tab`, [payload], config);
                successNotify("Service tab added successfully!");
            } else {
                // Update existing service tab
                response = await axios.patch(`${BASE_URL}/services-tab/${data.id}`, payload, config);
                successNotify("Service tab updated successfully!");
            }
            dispatch(GetAllServicesTabs())

            handleModalClose();
            setIsEdit(false);
            
            // Reset form
            setLabel("");
            setContent([]);
            setContentInput('');
            
            // Call success callback if provided
            if (onSuccess) {
                onSuccess();
            }

        } catch (error) {
            console.error('Service tab submission error:', error);
            const errorMessage = error.response?.data?.message || error.message || "Something went wrong!";
            errorNotify(errorMessage);
        } finally {
            setIsSubmitting(false);
        }
    }

    return (
        <fieldset className="reset mt-3 p-4 pt-0 t-0 ">
            <legend className="reset mb-3 side-heading px-5 mx-auto">{isEdit ? 'Update' : 'Add'} Service Tab</legend>
            <form onSubmit={addServiceTabHandler} >
                <Row className='mt-3'>
                    <Col xxl={12} className="">
                        <Row>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Service Tab Label</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setLabel(e.target.value)} 
                                        required={true} 
                                        label={"Enter Service Tab Label: *"} 
                                        type={"text"} 
                                        placeholder={"e.g., Mobile Development"} 
                                        controlId={"floatingInput-1"} 
                                        name="label" 
                                        value={label} 
                                    />
                                </div>
                            </Col>
                        </Row>
                        
                        <Row className='mt-3'>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Service Content Items</h5>
                                <div className='property-no'>
                                    <div className="mb-3">
                                        <label className="form-label fw-bold">Add Content Items *</label>
                                        <input
                                            type="text"
                                            className="form-control"
                                            value={contentInput}
                                            onChange={handleContentInputChange}
                                            onKeyPress={handleContentKeyPress}
                                            placeholder="Type content item and press Enter (e.g., React Native)"
                                        />
                                        <small className="text-muted">Press Enter to add each content item</small>
                                    </div>
                                    
                                    {content.length > 0 && (
                                        <div className="content-tags mt-3">
                                            <label className="form-label fw-bold">Content Items:</label>
                                            <div className="d-flex flex-wrap gap-2">
                                                {content.map((item, index) => (
                                                    <span
                                                        key={index}
                                                        className="badge bg-primary px-3 py-2 content-tag"
                                                        style={{ fontSize: '14px' }}
                                                    >
                                                        {item}
                                                        <button
                                                            type="button"
                                                            className="btn-close btn-close-white ms-2"
                                                            onClick={() => removeContent(item)}
                                                            style={{ fontSize: '8px' }}
                                                        ></button>
                                                    </span>
                                                ))}
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </Col>
                        </Row>



                        <Row>
                            <Col lg={12} className=' mt-5'>
                                <ButtonComp 
                                    type="submit" 
                                    variant={"outline-primary"} 
                                    className="btn-main-clr text-white" 
                                    btnText={isSubmitting ? 'Processing...' : (isEdit ? 'Update Service Tab' : 'Add Service Tab')} 
                                    disabled={isSubmitting}
                                />
                            </Col>
                        </Row>
                    </Col>
                </Row>
            </form>
        </fieldset>
    )
}

export default AddServiceTab