import { Col, Row } from 'react-bootstrap'
import { ButtonComp, InputComp } from '../../bootstrap'
import { useModal } from '../../../hooks/useModal'
import { useEffect, useState, useRef } from 'react'
import { useDispatch, useSelector } from 'react-redux'
import { errorNotify, successNotify } from '../../../Toastify/Toastify'
import axios from 'axios'
import { BASE_URL_AUTH } from '../../../config/service'
import { GetReviews } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice'
import { GetAllServicesPages } from '../../../store/slices/servicesSlice/servicesSlice'
import { Autocomplete, TextField, Chip, Switch, FormControlLabel } from '@mui/material'

const AddReview = ({ isEdit, data, setIsEdit, onSuccess }) => {
    console.log('[data]', data)
    const { handleModalClose } = useModal()
    const dispatch = useDispatch();
    const fileInputRef = useRef();

    // Review form states
    const [reviewerTitle, setReviewerTitle] = useState("")
    const [reviewerCompanyName, setReviewerCompanyName] = useState("")
    const [reviewText, setReviewText] = useState("")
    const [reviewerServiceName, setReviewerServiceName] = useState("")
    const [reviewerThumbnail, setReviewerThumbnail] = useState(null)
    const [fileName, setFileName] = useState('No file chosen')
    const [isSubmitting, setIsSubmitting] = useState(false)
    const [pages, setPages] = useState([])
    const [selectedPages, setSelectedPages] = useState([])
    const [showOnHome, setShowOnHome] = useState(false)

    useEffect(() => {
        dispatch(GetAllServicesPages())
            .then((response) => {
                if (response.payload.success) {
                    setPages(response.payload.data.map((item) => {
                        return {
                            label: item.page_heading,
                            value: item.route_url
                        }
                    }))
                }
            })
    }, [dispatch])

    useEffect(() => {
        if (isEdit && data) {
            setReviewerTitle(data.reviewer_title || "")
            setReviewerCompanyName(data.reviewer_company_name || "")
            setReviewText(data.review_text || "")
            setReviewerServiceName(data.reviewer_service_name || "")
            setReviewerThumbnail(data.reviewer_thumbnail || null)
            setFileName(data.reviewer_thumbnail ? 'Current image' : 'No file chosen')
            // Handle selected pages for edit mode if they exist
            if (data.pages.length > 0) {
                try {
                    setSelectedPages(data.pages.map(item => {
                        return {
                            label: item.page_heading,
                            value: item.route_url
                        }
                    }))
                } catch (e) {
                    setSelectedPages([])
                }
            } else {
                setSelectedPages([])
            }
            // Handle show_on_home for edit mode
            setShowOnHome(data.show_on_home || false)
        }
    }, [isEdit, data])

    const handleFileChange = (event) => {
        const file = event.target.files[0];
        if (file) {
            setReviewerThumbnail(file);
            setFileName(file.name);
        } else {
            setReviewerThumbnail(null);
            setFileName('No file chosen');
        }
    };

    const handleButtonClick = () => {
        fileInputRef.current.click();
    };

    const handlePageSelection = (selectedOptions) => {
        setSelectedPages(selectedOptions || []);
    };

    const addReviewHandler = async (e) => {
        e.preventDefault()
        setIsSubmitting(true)

        // Validation
        if (!reviewerTitle || !reviewerCompanyName || !reviewText || !reviewerServiceName || !reviewerThumbnail) {
            errorNotify("Required Fields are missing!")
            setIsSubmitting(false)
            return
        }

        console.log('selectedPages', )

        try {
            const formData = new FormData();
            formData.append('reviewer_title', reviewerTitle);
            formData.append('reviewer_company_name', reviewerCompanyName);
            formData.append('review_text', reviewText);
            formData.append('reviewer_service_name', reviewerServiceName);
            formData.append('reviewer_thumbnail', reviewerThumbnail);

            // Add selected pages as stringified array
            if (selectedPages.length > 0) {
                formData.append('pages', JSON.stringify(selectedPages.map(item => {
                    return { page_heading: item.label, route_url: item.value }
                })));
            }

            // Add show_on_home toggle value
            formData.append('show_on_home', showOnHome);

            // Get auth token from localStorage
            const token = localStorage.getItem('token');
            if (!token) {
                errorNotify("Authentication token not found!")
                setIsSubmitting(false)
                return
            }

            const config = {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'multipart/form-data'
                }
            };

            let response;
            if (!isEdit) {
                // Create new review
                response = await axios.post(`${BASE_URL_AUTH}/home/reviews`, formData, config);
                successNotify("Review added successfully!");

            } else {
                // Update existing review
                response = await axios.patch(`${BASE_URL_AUTH}/home/reviews/${data.id}`, formData, config);
                successNotify("Review updated successfully!");
            }
            dispatch(GetReviews())

            handleModalClose();
            setIsEdit(false);

            // Reset form
            setReviewerTitle("");
            setReviewerCompanyName("");
            setReviewText("");
            setReviewerServiceName("");
            setReviewerThumbnail(null);
            setFileName('No file chosen');
            setSelectedPages([]);
            setShowOnHome(false);

            // Call success callback if provided
            if (onSuccess) {
                onSuccess();
            }

        } catch (error) {
            console.error('Review submission error:', error);
            const errorMessage = error.response?.data?.message || error.message || "Something went wrong!";
            errorNotify(errorMessage);
        } finally {
            setIsSubmitting(false);
        }
    }

    return (
        <fieldset className="reset mt-3 p-4 pt-0 t-0 ">
            <legend className="reset mb-3 side-heading px-5 mx-auto">{isEdit ? 'Update' : 'Add'} Review</legend>
            <form onSubmit={addReviewHandler} >
                <Row className='mt-3'>
                    <Col xxl={12} className="">
                        <Row>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Reviewer Title</h5>
                                <div className='property-no'>
                                    <InputComp
                                        onChange={(e) => setReviewerTitle(e.target.value)}
                                        required={true}
                                        label={"Enter Reviewer Title: *"}
                                        type={"text"}
                                        placeholder={"e.g., CEO of Company"}
                                        controlId={"floatingInput-1"}
                                        name="reviewerTitle"
                                        value={reviewerTitle}
                                    />
                                </div>
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Company Name</h5>
                                <div className='property-no'>
                                    <InputComp
                                        onChange={(e) => setReviewerCompanyName(e.target.value)}
                                        required={true}
                                        label={"Enter Company Name: *"}
                                        type={"text"}
                                        placeholder={"Company Name"}
                                        controlId={"floatingInput-2"}
                                        name="reviewerCompanyName"
                                        value={reviewerCompanyName}
                                    />
                                </div>
                            </Col>
                        </Row>

                        <Row className='mt-3'>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Service Name</h5>
                                <div className='property-no'>
                                    <InputComp
                                        onChange={(e) => setReviewerServiceName(e.target.value)}
                                        required={true}
                                        label={"Enter Service Name: *"}
                                        type={"text"}
                                        placeholder={"e.g., Website Development"}
                                        controlId={"floatingInput-3"}
                                        name="reviewerServiceName"
                                        value={reviewerServiceName}
                                    />
                                </div>
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Review Text</h5>
                                <div className='property-no'>
                                    <InputComp
                                        onChange={(e) => setReviewText(e.target.value)}
                                        required={true}
                                        label={"Enter Review Text: *"}
                                        type={"text"}
                                        placeholder={"Review content"}
                                        controlId={"floatingInput-4"}
                                        name="reviewText"
                                        value={reviewText}
                                    />
                                </div>
                            </Col>
                        </Row>

                        <Row className='mt-3'>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Reviewer Thumbnail</h5>
                                <div className='file-upload_comp'>
                                    <input
                                        type="file"
                                        ref={fileInputRef}
                                        onChange={handleFileChange}
                                        accept="image/*"
                                        style={{ display: 'none' }}
                                    />
                                    <button
                                        type="button"
                                        onClick={handleButtonClick}
                                        className="btn btn-outline-primary"
                                    >
                                        Choose file
                                    </button>
                                    <span className="ms-2">{fileName}</span>
                                </div>
                                {!isEdit && (
                                    <small className="text-danger">* Reviewer thumbnail is required for new reviews</small>
                                )}
                            </Col>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Select pages (if any)</h5>
                                <div className='select-pages-dropdown'>
                                    <Autocomplete
                                        multiple
                                        options={pages?.length ? pages : []}
                                        value={selectedPages}
                                        onChange={(event, newValue) => {
                                            setSelectedPages(newValue || []);
                                        }}
                                        getOptionLabel={(option) => option.label}
                                        isOptionEqualToValue={(option, value) => option.value === value.value}
                                        renderInput={(params) => (
                                            <TextField
                                                {...params}
                                                variant="outlined"
                                                placeholder="Select multiple pages..."
                                                label="Select pages (if any)"
                                                size="small"
                                            />
                                        )}
                                        renderTags={(value, getTagProps) =>
                                            value.map((option, index) => (
                                                <Chip
                                                    label={option.label}
                                                    {...getTagProps({ index })}
                                                    size="small"
                                                    color="primary"
                                                    variant="outlined"
                                                />
                                            ))
                                        }
                                        noOptionsText="No pages available"
                                        clearOnBlur={false}
                                    />
                                </div>
                                {selectedPages.length > 0 && (
                                    <small className="text-muted mt-2 d-block">
                                        Selected: {selectedPages.map(page => page.label).join(', ')}
                                    </small>
                                )}
                                <div className='mt-3'>
                                    <FormControlLabel
                                        control={
                                            <Switch
                                                checked={showOnHome}
                                                onChange={(e) => setShowOnHome(e.target.checked)}
                                                color="primary"
                                            />
                                        }
                                        label="Show on Home"
                                    />
                                </div>
                            </Col>
                        </Row>

                        <Row>
                            <Col lg={12} className=' mt-5'>
                                <ButtonComp
                                    type="submit"
                                    variant={"outline-primary"}
                                    className="btn-main-clr text-white"
                                    btnText={isSubmitting ? 'Processing...' : (isEdit ? 'Update Review' : 'Add Review')}
                                    disabled={isSubmitting}
                                />
                            </Col>
                        </Row>
                    </Col>
                </Row>
            </form>
        </fieldset>
    )
}

export default AddReview