import { Col, Row } from 'react-bootstrap'
import { ButtonComp, InputComp, DropDownComp } from '../../bootstrap'
import { useModal } from '../../../hooks/useModal'
import { useEffect, useState, useRef } from 'react'
import { useDispatch } from 'react-redux'
import { errorNotify, successNotify } from '../../../Toastify/Toastify'
import axios from 'axios'
import { BASE_URL_AUTH } from '../../../config/service'
import { GetPromotions } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice'

const AddPromotion = ({ isEdit, data, setIsEdit, onSuccess }) => {
    console.log('[data]', data)
    const { handleModalClose } = useModal()
    const dispatch = useDispatch();
    const fileInputRef = useRef();

    // Promotion form states
    const [imgAlt, setImgAlt] = useState("")
    const [logoFor, setLogoFor] = useState("")
    const [imagePath, setImagePath] = useState(null)
    const [fileName, setFileName] = useState('No file chosen')
    const [isSubmitting, setIsSubmitting] = useState(false)

    // Logo options for dropdown
    const logoOptions = [
        { value: "Award_logo", label: "Awards" },
        { value: "Partners_logo", label: "Partners" }
    ]

    useEffect(() => {
        if (isEdit && data) {
            setImgAlt(data.img_alt || "")
            setLogoFor(data.logo_for || "")
            setFileName(data.image_path ? 'Current image' : 'No file chosen')
        }
    }, [isEdit, data])

    const handleFileChange = (event) => {
        const file = event.target.files[0];
        if (file) {
            setImagePath(file);
            setFileName(file.name);
        } else {
            setImagePath(null);
            setFileName('No file chosen');
        }
    };

    const handleButtonClick = () => {
        fileInputRef.current.click();
    };

    const addPromotionHandler = async (e) => {
        e.preventDefault()
        setIsSubmitting(true)

        // Validation
        if (!imgAlt || !logoFor || !imagePath) {
            errorNotify("Required Fields are missing!")
            setIsSubmitting(false)
            return
        }

        try {
            const formData = new FormData();
            formData.append('img_alt', imgAlt);
            formData.append('logo_for', logoFor);
            formData.append('image', imagePath);

            // Get auth token from localStorage
            const token = localStorage.getItem('token');
            if (!token) {
                errorNotify("Authentication token not found!")
                setIsSubmitting(false)
                return
            }

            const config = {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'multipart/form-data'
                }
            };

            let response;
            if (!isEdit) {
                // Create new promotion
                response = await axios.post(`${BASE_URL_AUTH}/logoslider`, formData, config);
                successNotify("Promotion added successfully!");

            } else {
                // Update existing promotion
                response = await axios.patch(`${BASE_URL_AUTH}/logoslider/${data.id}`, formData, config);
                successNotify("Promotion updated successfully!");
            }
            dispatch(GetPromotions())

            handleModalClose();
            setIsEdit(false);

            // Reset form
            setImgAlt("");
            setLogoFor("");
            setImagePath(null);
            setFileName('No file chosen');

            // Call success callback if provided
            if (onSuccess) {
                onSuccess();
            }

        } catch (error) {
            console.error('Promotion submission error:', error);
            const errorMessage = error.response?.data?.message || error.message || "Something went wrong!";
            errorNotify(errorMessage);
        } finally {
            setIsSubmitting(false);
        }
    }

    return (
        <fieldset className="reset mt-3 p-4 pt-0 t-0 ">
            <legend className="reset mb-3 side-heading px-5 mx-auto">{isEdit ? 'Update' : 'Add'} Promotion</legend>
            <form onSubmit={addPromotionHandler} >
                <Row className='mt-3'>
                    <Col xxl={12} className="">
                        <Row>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Logo For</h5>
                                <div className='property-no'>
                                    <DropDownComp
                                        onChange={(e) => setLogoFor(e.target.value)}
                                        required={true}
                                        label={"Select Logo Type: *"}
                                        controlId={"floatingSelect-1"}
                                        name="logoFor"
                                        value={logoFor}
                                        options={logoOptions}
                                        placeholder="Select logo type"
                                    />
                                </div>
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Image Alt Text</h5>
                                <div className='property-no'>
                                    <InputComp
                                        onChange={(e) => setImgAlt(e.target.value)}
                                        required={true}
                                        label={"Enter Image Alt Text: *"}
                                        type={"text"}
                                        placeholder={"Image Alt Text"}
                                        controlId={"floatingInput-1"}
                                        name="imgAlt"
                                        value={imgAlt}
                                    />
                                </div>
                            </Col>
                        </Row>

                        <Row className='mt-3'>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Promotion Image</h5>
                                <div className='file-upload_comp'>
                                    <input
                                        type="file"
                                        ref={fileInputRef}
                                        onChange={handleFileChange}
                                        accept="image/*"
                                        style={{ display: 'none' }}
                                    />
                                    <button
                                        type="button"
                                        onClick={handleButtonClick}
                                        className="btn btn-outline-primary"
                                    >
                                        Choose file
                                    </button>
                                    <span className="ms-2">{fileName}</span>
                                </div>
                                {!isEdit && (
                                    <small className="text-danger">* Promotion image is required for new promotions</small>
                                )}
                            </Col>
                        </Row>

                        <Row>
                            <Col lg={12} className=' mt-5'>
                                <ButtonComp
                                    type="submit"
                                    variant={"outline-primary"}
                                    className="btn-main-clr text-white"
                                    btnText={isSubmitting ? 'Processing...' : (isEdit ? 'Update Promotion' : 'Add Promotion')}
                                    disabled={isSubmitting}
                                />
                            </Col>
                        </Row>
                    </Col>
                </Row>
            </form>
        </fieldset>
    )
}

export default AddPromotion