import { Col, Row } from 'react-bootstrap'
import { ButtonComp, InputComp } from '../../bootstrap'
import { useModal } from '../../../hooks/useModal'
import { useEffect, useState, useRef } from 'react'
import { useDispatch } from 'react-redux'
import { errorNotify, successNotify } from '../../../Toastify/Toastify'
import axios from 'axios'
import { BASE_URL } from '../../../config/service'
import { GetHomePageBanners, GetPortfolioListing } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice'

const AddPortfolio = ({ isEdit, data, setIsEdit, onSuccess }) => {
    console.log('[data]',data)
    const { handleModalClose } = useModal()
    const dispatch = useDispatch();
    const fileInputRef = useRef();
    
    // Portfolio form states
    const [itemHeading, setItemHeading] = useState("")
    const [itemLink, setItemLink] = useState("")
    const [mediaAlt, setMediaAlt] = useState("")
    const [media, setMedia] = useState(null)
    const [fileName, setFileName] = useState('No file chosen')
    const [isSubmitting, setIsSubmitting] = useState(false)

    useEffect(() => {
        if (isEdit && data) {
            setItemHeading(data.item_heading || "")
            setItemLink(data.item_link || "")
            setMediaAlt(data.media_alt || "")
            setFileName(data.media ? 'Current image' : 'No file chosen')
        }
    }, [isEdit, data])

    const handleFileChange = (event) => {
        const file = event.target.files[0];
        if (file) {
            setMedia(file);
            setFileName(file.name);
        } else {
            setMedia(null);
            setFileName('No file chosen');
        }
    };

    const handleButtonClick = () => {
        fileInputRef.current.click();
    };

    const addPortfolioHandler = async (e) => {
        e.preventDefault()
        setIsSubmitting(true)

        // Validation
        if (!itemHeading || !itemLink || !mediaAlt) {
            errorNotify("Required Fields are missing!")
            setIsSubmitting(false)
            return
        }

        // Validate media file for new portfolio items
        if (!isEdit && !media) {
            errorNotify("Portfolio image is required!")
            setIsSubmitting(false)
            return
        }

        try {
            const formData = new FormData();
            formData.append('item_heading', itemHeading);
            formData.append('item_link', itemLink);
            formData.append('media_alt', mediaAlt);
            
            if (media) {
                formData.append('media', media);
            }

            // Get auth token from localStorage
            const token = localStorage.getItem('token');
            if (!token) {
                errorNotify("Authentication token not found!")
                setIsSubmitting(false)
                return
            }

            const config = {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'multipart/form-data'
                }
            };

            let response;
            if (!isEdit) {
                // Create new portfolio item
                response = await axios.post(`${BASE_URL}/portfolio-item`, formData, config);
                successNotify("Portfolio item added successfully!");
            } else {
                // Update existing portfolio item
                response = await axios.put(`${BASE_URL}/portfolio-item/${data.id}`, formData, config);
                successNotify("Portfolio item updated successfully!");
            }

            dispatch(GetPortfolioListing())

            handleModalClose();
            setIsEdit(false);
            
            // Reset form
            setItemHeading("");
            setItemLink("");
            setMediaAlt("");
            setMedia(null);
            setFileName('No file chosen');
            
            // Call success callback if provided
            if (onSuccess) {
                onSuccess();
            }

        } catch (error) {
            console.error('Portfolio submission error:', error);
            const errorMessage = error.response?.data?.message || error.message || "Something went wrong!";
            errorNotify(errorMessage);
        } finally {
            setIsSubmitting(false);
        }
    }

    return (
        <fieldset className="reset mt-3 p-4 pt-0 t-0 ">
            <legend className="reset mb-3 side-heading px-5 mx-auto">{isEdit ? 'Update' : 'Add'} Portfolio Item</legend>
            <form onSubmit={addPortfolioHandler} >
                <Row className='mt-3'>
                    <Col xxl={12} className="">
                        <Row>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Item Heading</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setItemHeading(e.target.value)} 
                                        required={true} 
                                        label={"Enter Item Heading: *"} 
                                        type={"text"} 
                                        placeholder={"Item Heading"} 
                                        controlId={"floatingInput-1"} 
                                        name="itemHeading" 
                                        value={itemHeading} 
                                    />
                                </div>
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Item Link</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setItemLink(e.target.value)} 
                                        required={true} 
                                        label={"Enter Item Link: *"} 
                                        type={"url"} 
                                        placeholder={"https://example.com"} 
                                        controlId={"floatingInput-2"} 
                                        name="itemLink" 
                                        value={itemLink} 
                                    />
                                </div>
                            </Col>
                        </Row>

                        <Row className='mt-3'>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Portfolio Image</h5>
                                <div className='file-upload_comp'>
                                    <input
                                        type="file"
                                        ref={fileInputRef}
                                        onChange={handleFileChange}
                                        accept="image/*"
                                        style={{ display: 'none' }}
                                    />
                                    <button 
                                        type="button" 
                                        onClick={handleButtonClick}
                                        className="btn btn-outline-primary"
                                    >
                                        Choose file
                                    </button>
                                    <span className="ms-2">{fileName}</span>
                                </div>
                                {!isEdit && (
                                    <small className="text-danger">* Portfolio image is required for new items</small>
                                )}
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Image Alt Text</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setMediaAlt(e.target.value)} 
                                        required={true} 
                                        label={"Enter Image Alt Text: *"} 
                                        type={"text"} 
                                        placeholder={"Image Alt Text"} 
                                        controlId={"floatingInput-3"} 
                                        name="mediaAlt" 
                                        value={mediaAlt} 
                                    />
                                </div>
                            </Col>
                        </Row>

                        <Row>
                            <Col lg={12} className=' mt-5'>
                                <ButtonComp 
                                    type="submit" 
                                    variant={"outline-primary"} 
                                    className="btn-main-clr text-white" 
                                    btnText={isSubmitting ? 'Processing...' : (isEdit ? 'Update Portfolio Item' : 'Add Portfolio Item')} 
                                    disabled={isSubmitting}
                                />
                            </Col>
                        </Row>
                    </Col>
                </Row>
            </form>
        </fieldset>
    )
}

export default AddPortfolio