import { Col, Row } from 'react-bootstrap'
import { ButtonComp, InputComp } from '../../bootstrap'
import { useModal } from '../../../hooks/useModal'
import { useEffect, useState } from 'react'
import { useDispatch } from 'react-redux'
import { errorNotify, successNotify } from '../../../Toastify/Toastify'
import axios from 'axios'
import { BASE_URL } from '../../../config/service'
import { GetAllFaqs } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice'
import { GetAllServicesPages } from '../../../store/slices/servicesSlice/servicesSlice'
import { Autocomplete, TextField, Chip, Switch, FormControlLabel } from '@mui/material'

const AddFaq = ({ isEdit, data, setIsEdit, onSuccess }) => {
    const { handleModalClose } = useModal()
    const dispatch = useDispatch();
    const [question, setQuestion] = useState("")
    const [answer, setAnswer] = useState("")
    const [isSubmitting, setIsSubmitting] = useState(false)
    const [pages, setPages] = useState([])
    const [selectedPages, setSelectedPages] = useState([])
    const [showOnHome, setShowOnHome] = useState(false)

    useEffect(() => {
        dispatch(GetAllServicesPages())
            .then((response) => {
                if (response.payload.success) {
                    setPages(response.payload.data.map((item) => {
                        return {
                            label: item.page_heading,
                            value: item.route_url
                        }
                    }))
                }
            })
    }, [dispatch])

    useEffect(() => {
        if (isEdit && data) {
            setQuestion(data.question || "")
            setAnswer(data.answer || "")
            // Handle selected pages for edit mode if they exist
            if (data.pages) {
                try {
                    const parsedPages = JSON.parse(data.pages)
                    setSelectedPages(parsedPages.map(item => {
                        return {
                            label: item.page_heading,
                            value: item.route_url
                        }
                    }))
                } catch (e) {
                    setSelectedPages([])
                }
            } else {
                setSelectedPages([])
            }
            // Handle show_on_home for edit mode
            setShowOnHome(data.show_on_home || false)
        }
    }, [isEdit, data])

    const addFaqHandler = async (e) => {
        e.preventDefault()
        setIsSubmitting(true)

        if (!question || !answer) {
            errorNotify("Required Fields are missing!")
            setIsSubmitting(false)
            return
        }

        try {
            const payload = {
                question: question,
                answer: answer,
                show_on_home: showOnHome
            }

            // Add selected pages if any
            if (selectedPages.length > 0) {
                payload.pages = JSON.stringify(selectedPages.map(item => {
                    return { page_heading: item.label, route_url: item.value }
                }))
            }

            // Get auth token from localStorage
            const token = localStorage.getItem('token');
            if (!token) {
                errorNotify("Authentication token not found!")
                setIsSubmitting(false)
                return
            }

            const config = {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json'
                }
            };

            let response;
            if (!isEdit) {
                // Create new FAQ
                response = await axios.post(`${BASE_URL}/home/faqs`, payload, config);
                successNotify("FAQ added successfully!");
            } else {
                // Update existing FAQ
                response = await axios.patch(`${BASE_URL}/home/faqs/${data.id}`, payload, config);
                successNotify("FAQ updated successfully!");
            }

            dispatch(GetAllFaqs())

            handleModalClose();
            setIsEdit(false);
            
            // Reset form
            setQuestion("");
            setAnswer("");
            setSelectedPages([]);
            setShowOnHome(false);
            
            // Call success callback if provided
            if (onSuccess) {
                onSuccess();
            }

        } catch (error) {
            console.error('FAQ submission error:', error);
            const errorMessage = error.response?.data?.message || error.message || "Something went wrong!";
            errorNotify(errorMessage);
        } finally {
            setIsSubmitting(false);
        }
    }

    return (
        <fieldset className="reset mt-3 p-4 pt-0 t-0 ">
            <legend className="reset mb-3 side-heading px-5 mx-auto">{isEdit ? 'Update' : 'Add'} FAQ</legend>
            <form onSubmit={addFaqHandler} >
                <Row className='mt-3'>
                    <Col xxl={12} className="">
                        <Row>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Question</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setQuestion(e.target.value)} 
                                        required={true} 
                                        label={"Enter Question: *"} 
                                        type={"text"} 
                                        placeholder={"Enter your question here"} 
                                        controlId={"floatingInput-1"} 
                                        name="question" 
                                        value={question} 
                                    />
                                </div>
                            </Col>
                        </Row>
                        <Row className='mt-3'>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Answer</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setAnswer(e.target.value)} 
                                        required={true} 
                                        label={"Enter Answer: *"} 
                                        type={"text"} 
                                        placeholder={"Enter your answer here"} 
                                        controlId={"floatingInput-2"} 
                                        name="answer" 
                                        value={answer} 
                                    />
                                </div>
                            </Col>
                        </Row>
                        <Row className='mt-3'>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Select pages (if any)</h5>
                                <div className='select-pages-dropdown'>
                                    <Autocomplete
                                        multiple
                                        options={pages?.length ? pages : []}
                                        value={selectedPages}
                                        onChange={(event, newValue) => {
                                            setSelectedPages(newValue || []);
                                        }}
                                        getOptionLabel={(option) => option.label}
                                        isOptionEqualToValue={(option, value) => option.value === value.value}
                                        renderInput={(params) => (
                                            <TextField
                                                {...params}
                                                variant="outlined"
                                                placeholder="Select multiple pages..."
                                                label="Select pages (if any)"
                                                size="small"
                                            />
                                        )}
                                        renderTags={(value, getTagProps) =>
                                            value.map((option, index) => (
                                                <Chip
                                                    label={option.label}
                                                    {...getTagProps({ index })}
                                                    size="small"
                                                    color="primary"
                                                    variant="outlined"
                                                />
                                            ))
                                        }
                                        noOptionsText="No pages available"
                                        clearOnBlur={false}
                                    />
                                </div>
                                {selectedPages.length > 0 && (
                                    <small className="text-muted mt-2 d-block">
                                        Selected: {selectedPages.map(page => page.label).join(', ')}
                                    </small>
                                )}
                                <div className='mt-3'>
                                    <FormControlLabel
                                        control={
                                            <Switch
                                                checked={showOnHome}
                                                onChange={(e) => setShowOnHome(e.target.checked)}
                                                color="primary"
                                            />
                                        }
                                        label="Show on Home"
                                    />
                                </div>
                            </Col>
                        </Row>
                        <Row>
                            <Col lg={12} className=' mt-5'>
                                <ButtonComp 
                                    type="submit" 
                                    variant={"outline-primary"} 
                                    className="btn-main-clr text-white" 
                                    btnText={isSubmitting ? 'Processing...' : (isEdit ? 'Update FAQ' : 'Add FAQ')} 
                                    disabled={isSubmitting}
                                />
                            </Col>
                        </Row>
                    </Col>
                </Row>
            </form>
        </fieldset>
    )
}

export default AddFaq    