import React, { useState, useRef, useEffect } from 'react';
import { Container, Row, Col, Card, Button, Form, Alert } from 'react-bootstrap';
import ReactQuill from 'react-quill';
import 'react-quill/dist/quill.snow.css';
import InputComp from '../../bootstrap/InputComp';
import DropDownComp from '../../bootstrap/DropDownComp';
import { useFile } from '../../../hooks/useFile';
import PlaceholderImg from "../../../assets/img/placeholder-img.jpg";
import './AddBlogPost.css';
import { GetBlogPostById, GetCategories } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { useDispatch, useSelector } from 'react-redux';
import axios from 'axios';
import { useLocation, useNavigate, useParams } from 'react-router-dom';

const AddBlogPost = () => {
  const [formData, setFormData] = useState({
    title: '',
    slug: '',
    image_alt: '',
    category_id: '',
    is_featured: '',
    is_active: '',
    content: ''
  });

  const dispatch = useDispatch()
  const { id } = useParams()
  const { pathname } = useLocation()

  const [featuredImage, setFeaturedImage] = useState(null);
  const [imagePreview, setImagePreview] = useState(PlaceholderImg);
  const [categories, setCategories] = useState([]);
  const [loading, setLoading] = useState(false);
  const [message, setMessage] = useState({ type: '', text: '' });
  const [isLoadingDetails, setIsLoadingDetails] = useState(false);
  const fileInputRef = useRef();
  const { handleFileChange: originalHandleFileChange, fileName } = useFile();
  const { blogPostDetails } = useSelector(state => state.PropertyMangementReducer)
  console.log('[blogPostDetails]', blogPostDetails)
  const navigate = useNavigate()

  const yesNoOptions = [
    { label: 'Yes', value: '1' },
    { label: 'No', value: '0' }
  ];

  const isEditPath = pathname.includes('/edit/')


  

  useEffect(() => {
    if (isEditPath && id) {
      setIsLoadingDetails(true);
      dispatch(GetBlogPostById(id))
        .then(() => {
          setIsLoadingDetails(false);
        })
        .catch(() => {
          setIsLoadingDetails(false);
        });
    }
  }, [id, isEditPath, dispatch])

  // Fetch categories on component mount
  useEffect(() => {
    dispatch(GetCategories())
      .then((res) => {
        const values = res.payload
        setCategories(values.map((item) => {
          return {
            label: item.name,
            value: item.id
          }
        }))
      })
  }, [dispatch]);

  // Prefill form data when blogPostDetails is available (for edit mode)
  useEffect(() => {
    if (blogPostDetails && isEditPath) {
      setFormData({
        title: blogPostDetails.title || '',
        slug: blogPostDetails.slug || '',
        image_alt: blogPostDetails.image_alt || '',
        category_id: blogPostDetails.category_id?.toString() || '',
        is_featured: blogPostDetails.is_featured ? '1' : '0',
        is_active: blogPostDetails.is_active ? '1' : '0',
        content: blogPostDetails.content || ''
      });
      
      // Set image preview if feature_image exists
      if (blogPostDetails.feature_image) {
        setImagePreview(blogPostDetails.feature_image);
      }
    }
  }, [blogPostDetails, isEditPath]);


  const handleInputChange = (e) => {
    const { name, value } = e.target;
    setFormData(prev => ({
      ...prev,
      [name]: value
    }));
  };

  const handleSlugGeneration = () => {
    const slug = formData.title
      .toLowerCase()
      .replace(/[^a-z0-9 -]/g, '')
      .replace(/\s+/g, '-')
      .replace(/-+/g, '-')
      .trim('-');

    setFormData(prev => ({
      ...prev,
      slug: `/${slug}`
    }));
  };

  const handleFileChange = (e) => {
    const file = e.target.files[0];
    if (file) {
      setFeaturedImage(file);
      const reader = new FileReader();
      reader.onload = (e) => {
        setImagePreview(e.target.result);
      };
      reader.readAsDataURL(file);
    }
  };

  const handleButtonClick = () => {
    fileInputRef.current.click();
  };

  const handleContentChange = (content) => {
    setFormData(prev => ({
      ...prev,
      content
    }));
  };

  const resetForm = () => {
    // if (isEditPath && blogPostDetails) {
    //   // Reset to original data for edit mode
    //   setFormData({
    //     title: blogPostDetails.title || '',
    //     slug: blogPostDetails.slug || '',
    //     image_alt: blogPostDetails.image_alt || '',
    //     category_id: blogPostDetails.category_id?.toString() || '',
    //     is_featured: blogPostDetails.is_featured ? '1' : '0',
    //     is_active: blogPostDetails.is_active ? '1' : '0',
    //     content: blogPostDetails.content || ''
    //   });
    //   setImagePreview(blogPostDetails.feature_image || PlaceholderImg);
    // } else {
    //   // Reset to empty for add mode
    //   setFormData({
    //     title: '',
    //     slug: '',
    //     image_alt: '',
    //     category_id: '',
    //     is_featured: '',
    //     is_active: '',
    //     content: ''
    //   });
    //   setFeaturedImage(null);
    //   setImagePreview(PlaceholderImg);
    // }
    // setMessage({ type: '', text: '' });
    navigate('/dashboard/blog-post')
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);
    setMessage({ type: '', text: '' });

    // Validate required fields
    if (!formData.title || !formData.slug || !formData.category_id || !formData.content) {
      setMessage({ type: 'error', text: 'Please fill in all required fields.' });
      setLoading(false);
      return;
    }

    const formDataToSend = new FormData();
    formDataToSend.append('title', formData.title);
    formDataToSend.append('slug', formData.slug);
    formDataToSend.append('image_alt', formData.image_alt);
    formDataToSend.append('category_id', formData.category_id);
    formDataToSend.append('is_featured', formData.is_featured || '0');
    formDataToSend.append('is_active', formData.is_active || '1');
    formDataToSend.append('content', formData.content);

    if (featuredImage) {
      formDataToSend.append('feature_image', featuredImage);
    }


    try {
      let response;
      const config = {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`,
          'Content-Type': 'multipart/form-data'
        }
      };

      if (isEditPath && id) {
        // Update existing blog post
        response = await axios.patch(`http://localhost:3000/api/posts/${id}`, formDataToSend, config);
        if (response.status === 200) {
          setMessage({ type: 'success', text: 'Blog post updated successfully!' });
        }
      } else {
        // Create new blog post
        response = await axios.post('http://localhost:3000/api/posts', formDataToSend, config);
        if (response.status === 200) {
          setMessage({ type: 'success', text: 'Blog post created successfully!' });
          resetForm();
        }
      }

      if (response.status !== 200) {
        setMessage({ type: 'error', text: response.data.message || 'Error processing blog post' });
      }
    } catch (error) {
      console.error('Error:', error);
      setMessage({ type: 'error', text: error.response?.data?.message || 'Network error. Please try again.' });
    } finally {
      setLoading(false);
    }
  };

  const quillModules = {
    toolbar: [
      [{ 'header': [1, 2, 3, false] }],
      ['bold', 'italic', 'underline', 'strike'],
      [{ 'list': 'ordered' }, { 'list': 'bullet' }],
      [{ 'color': [] }, { 'background': [] }],
      [{ 'align': [] }],
      ['link', 'image'],
      ['clean']
    ],
  };

  const quillFormats = [
    'header',
    'bold', 'italic', 'underline', 'strike',
    'list', 'bullet',
    'color', 'background',
    'align',
    'link', 'image'
  ];

  return (
    <Container fluid className="add-blog-post">
      <Row>
        <Col>
          <Card className="shadow-sm">
            <Card.Header className="text-white" style={{ backgroundColor: '#16BFBF' }}>
              <h4 className="mb-0">{isEditPath ? 'Edit Blog Post' : 'Add New Blog Post'}</h4>
            </Card.Header>
            <Card.Body>
              {message.text && (
                <Alert variant={message.type === 'success' ? 'success' : 'danger'} dismissible>
                  {message.text}
                </Alert>
              )}

              {isLoadingDetails && isEditPath && (
                <div className="text-center py-4">
                  <div className="spinner-border text-primary" role="status">
                    <span className="visually-hidden">Loading...</span>
                  </div>
                  <p className="mt-2">Loading blog post details...</p>
                </div>
              )}
              
              <Form onSubmit={handleSubmit} className={loading || isLoadingDetails ? 'loading' : ''}>
                <Row>
                  <Col md={8}>
                    {/* Title */}
                    <label className="form-label">Blog Post Title *</label>
                    <InputComp
                      label="Blog Post Title *"
                      type="text"
                      placeholder="Enter blog post title"
                      name="title"
                      value={formData.title}
                      onChange={handleInputChange}
                      required
                    />

                    {/* Slug */}
                    <Row>
                      <Col md={8}>
                        <label className="form-label">Slug *</label>
                        <InputComp
                          label="Slug *"
                          type="text"
                          placeholder="Enter slug"
                          name="slug"
                          value={formData.slug}
                          onChange={handleInputChange}
                          required
                        />
                      </Col>
                    </Row>

                    <Row>
                      {/* Content Editor */}
                      <div className="mb-4">
                        <label className="form-label">Content *</label>
                        <div className="quill-editor-container">
                          <ReactQuill
                            theme="snow"
                            value={formData.content}
                            onChange={handleContentChange}
                            modules={quillModules}
                            formats={quillFormats}
                            placeholder="Write your blog content here..."
                            style={{ height: '400px' }}
                          />
                        </div>
                      </div>
                    </Row>

                    {/* Featured Image */}


                    {/* Image Alt */}


                    {/* Category Dropdown */}
                    <div className="mb-4">
                      <label className="form-label">Category *</label>
                      <DropDownComp
                        name="category_id"
                        value={formData.category_id}
                        onChange={handleInputChange}
                        options={categories}
                        label="Select Category"
                        className="form-select"
                      />
                    </div>

                    {/* Is Featured Dropdown */}
                    <div className="mb-4">
                      <label className="form-label">Featured</label>
                      <DropDownComp
                        name="is_featured"
                        value={formData.is_featured}
                        onChange={handleInputChange}
                        options={yesNoOptions}
                        label="Select Option"
                        className="form-select"
                      />
                    </div>

                    {/* Is Active Dropdown */}
                    <div className="mb-4">
                      <label className="form-label">Active</label>
                      <DropDownComp
                        name="is_active"
                        value={formData.is_active}
                        onChange={handleInputChange}
                        options={yesNoOptions}
                        label="Select Option"
                        className="form-select"
                      />
                    </div>
                  </Col>

                  <Col md={4}>


                    <div className="mb-4">
                      <label className="form-label">Featured Image</label>
                      <div className="file-upload-section d-flex align-items-center gap-3 py-3">
                        <div className="file-attach-container flex-grow-1">
                          <fieldset className='reset'>
                            <div className="file-upload_comp">
                              <input
                                type="file"
                                ref={fileInputRef}
                                onChange={handleFileChange}
                                accept="image/*"
                                style={{ display: 'none' }}
                              />
                              <button
                                type="button"
                                onClick={handleButtonClick}
                                className="btn btn-outline-primary"
                              >
                                Choose file
                              </button>
                              <span className="ms-2">{fileName}</span>
                            </div>
                          </fieldset>
                        </div>
                        <div className="sample-attach-img flex-grow-1 mt-2">
                          <img
                            src={imagePreview}
                            alt="Preview"
                            width={120}
                            height={75}
                            className="img-thumbnail"
                          />
                        </div>
                      </div>
                    </div>
                    <div>
                      <label className="form-label">Image Alt Text</label>
                      <InputComp
                        label="Image Alt Text"
                        type="text"
                        placeholder="Enter image alt text"
                        name="image_alt"
                        value={formData.image_alt}
                        onChange={handleInputChange}
                      />
                    </div>
                  </Col>
                </Row>

                {/* Submit Button */}
                <Row>
                  <Col>
                    <div className="d-flex gap-2 justify-content-end">
                                            <Button 
                        variant="secondary" 
                        type="button"
                        onClick={resetForm}
                        disabled={loading || isLoadingDetails}
                      >
                        Cancel
                      </Button>
                      <Button 
                        variant="primary" 
                        type="submit"
                        disabled={loading || isLoadingDetails}
                      >
                        {loading ? (isEditPath ? 'Updating...' : 'Publishing...') : (isEditPath ? 'Update Blog Post' : 'Publish Blog Post')}
                      </Button>
                    </div>
                  </Col>
                </Row>
              </Form>
            </Card.Body>
          </Card>
        </Col>
      </Row>
    </Container>
  );
};

export default AddBlogPost;