import { Col, Row } from 'react-bootstrap'
import { ButtonComp, InputComp } from '../../bootstrap'
import { useModal } from '../../../hooks/useModal'
import { useEffect, useState, useRef } from 'react'
import { useDispatch } from 'react-redux'
import { errorNotify, successNotify } from '../../../Toastify/Toastify'
import axios from 'axios'
import { BASE_URL } from '../../../config/service'
import { GetHomePageBanners } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice'

const AddBanner = ({ isEdit, data, setIsEdit, onSuccess }) => {
    console.log('[data]',data)
    const { handleModalClose } = useModal()
    const dispatch = useDispatch();
    const fileInputRef = useRef();
    
    // Banner form states
    const [mainHeading, setMainHeading] = useState("")
    const [content, setContent] = useState("")
    const [buttonText, setButtonText] = useState("")
    const [buttonLink, setButtonLink] = useState("")
    const [imgAlt, setImgAlt] = useState("")
    const [bannerImg, setBannerImg] = useState(null)
    const [fileName, setFileName] = useState('No file chosen')
    const [isSubmitting, setIsSubmitting] = useState(false)
    const [showButton, setShowButton] = useState(true)

    useEffect(() => {
        if (isEdit && data) {
            setMainHeading(data.main_heading || "")
            setContent(data.content || "")
            setButtonText(data.button_text || "")
            setButtonLink(data.button_link || "")
            setImgAlt(data.img_alt || "")
            setFileName(data.banner_img ? 'Current image' : 'No file chosen')
            // Show button fields if button data exists
            setShowButton(!!(data.button_text || data.button_link))
        }
    }, [isEdit, data])

    const handleFileChange = (event) => {
        const file = event.target.files[0];
        if (file) {
            setBannerImg(file);
            setFileName(file.name);
        } else {
            setBannerImg(null);
            setFileName('No file chosen');
        }
    };

    const handleButtonClick = () => {
        fileInputRef.current.click();
    };

    const addBannerHandler = async (e) => {
        e.preventDefault()
        setIsSubmitting(true)

        // Validation
        if (!mainHeading || !content || !imgAlt) {
            errorNotify("Required Fields are missing!")
            setIsSubmitting(false)
            return
        }

        // Validate button fields if showButton is true
        if (showButton && (!buttonText || !buttonLink)) {
            errorNotify("Button text and link are required when button is enabled!")
            setIsSubmitting(false)
            return
        }

        try {
            const formData = new FormData();
            formData.append('main_heading', mainHeading);
            formData.append('content', content);
            formData.append('img_alt', imgAlt);
            
            // Only append button fields if showButton is true
            if (showButton) {
                formData.append('button_text', buttonText);
                formData.append('button_link', buttonLink);
            }
            
            if (bannerImg) {
                formData.append('banner_img', bannerImg);
            }

            // Get auth token from localStorage
            const token = localStorage.getItem('token');
            if (!token) {
                errorNotify("Authentication token not found!")
                setIsSubmitting(false)
                return
            }

            const config = {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'multipart/form-data'
                }
            };

            let response;
            if (!isEdit) {
                // Create new banner
                response = await axios.post(`${BASE_URL}/home/slider-banner`, formData, config);
                successNotify("Banner added successfully!");
                dispatch(GetHomePageBanners())
            } else {
                // Update existing banner
                response = await axios.patch(`${BASE_URL}/home/slider-banner/${data.id}`, formData, config);
                successNotify("Banner updated successfully!");
                dispatch(GetHomePageBanners())
            }

            handleModalClose();
            setIsEdit(false);
            
            // Reset form
            setMainHeading("");
            setContent("");
            setButtonText("");
            setButtonLink("");
            setImgAlt("");
            setBannerImg(null);
            setFileName('No file chosen');
            setShowButton(false);
            
            // Call success callback if provided
            if (onSuccess) {
                onSuccess();
            }

        } catch (error) {
            console.error('Banner submission error:', error);
            const errorMessage = error.response?.data?.message || error.message || "Something went wrong!";
            errorNotify(errorMessage);
        } finally {
            setIsSubmitting(false);
        }
    }

    return (
        <fieldset className="reset mt-3 p-4 pt-0 t-0 ">
            <legend className="reset mb-3 side-heading px-5 mx-auto">{isEdit ? 'Update' : 'Add'} Banner</legend>
            <form onSubmit={addBannerHandler} >
                <Row className='mt-3'>
                    <Col xxl={12} className="">
                        <Row>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Main Heading</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setMainHeading(e.target.value)} 
                                        required={true} 
                                        label={"Enter Main Heading: *"} 
                                        type={"text"} 
                                        placeholder={"Main Heading"} 
                                        controlId={"floatingInput-1"} 
                                        name="mainHeading" 
                                        value={mainHeading} 
                                    />
                                </div>
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Banner Content</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setContent(e.target.value)} 
                                        required={true} 
                                        label={"Enter Content: *"} 
                                        type={"text"} 
                                        placeholder={"Content"} 
                                        controlId={"floatingInput-2"} 
                                        name="content" 
                                        value={content} 
                                    />
                                </div>
                            </Col>
                        </Row>
                        
                        <Row className='mt-3'>
                            <Col lg={12} className='bg-lightgray p-3'>
                                <div className="form-check">
                                    <input
                                        className="form-check-input"
                                        type="checkbox"
                                        id="showButtonCheckbox"
                                        checked={showButton}
                                        onChange={(e) => setShowButton(e.target.checked)}
                                    />
                                    <label className="form-check-label" htmlFor="showButtonCheckbox">
                                        <strong>Add Button to Banner</strong>
                                    </label>
                                </div>
                            </Col>
                        </Row>

                        {showButton && (
                            <Row className='mt-3'>
                                <Col lg={6} className='bg-lightgray p-3'>
                                    <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Button Text</h5>
                                    <div className='property-no'>
                                        <InputComp 
                                            onChange={(e) => setButtonText(e.target.value)} 
                                            required={true} 
                                            label={"Enter Button Text: *"} 
                                            type={"text"} 
                                            placeholder={"Button Text"} 
                                            controlId={"floatingInput-3"} 
                                            name="buttonText" 
                                            value={buttonText} 
                                        />
                                    </div>
                                </Col>
                                <Col lg={6} className='bg-lightgray p-3'>
                                    <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Button Link</h5>
                                    <div className='property-no'>
                                        <InputComp 
                                            onChange={(e) => setButtonLink(e.target.value)} 
                                            required={true} 
                                            label={"Enter Button Link: *"} 
                                            type={"url"} 
                                            placeholder={"https://example.com"} 
                                            controlId={"floatingInput-4"} 
                                            name="buttonLink" 
                                            value={buttonLink} 
                                        />
                                    </div>
                                </Col>
                            </Row>
                        )}

                        <Row className='mt-3'>
                            
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Banner Image</h5>
                                <div className='file-upload_comp'>
                                    <input
                                        type="file"
                                        ref={fileInputRef}
                                        onChange={handleFileChange}
                                        accept="image/*"
                                        style={{ display: 'none' }}
                                    />
                                    <button 
                                        type="button" 
                                        onClick={handleButtonClick}
                                        className="btn btn-outline-primary"
                                    >
                                        Choose file
                                    </button>
                                    <span className="ms-2">{fileName}</span>
                                </div>
                                {!isEdit && (
                                    <small className="text-danger">* Banner image is required for new banners</small>
                                )}
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Image Alt Text</h5>
                                <div className='property-no'>
                                    <InputComp 
                                        onChange={(e) => setImgAlt(e.target.value)} 
                                        required={true} 
                                        label={"Enter Image Alt Text: *"} 
                                        type={"text"} 
                                        placeholder={"Image Alt Text"} 
                                        controlId={"floatingInput-5"} 
                                        name="imgAlt" 
                                        value={imgAlt} 
                                    />
                                </div>
                            </Col>
                        </Row>

                        <Row>
                            <Col lg={12} className=' mt-5'>
                                <ButtonComp 
                                    type="submit" 
                                    variant={"outline-primary"} 
                                    className="btn-main-clr text-white" 
                                    btnText={isSubmitting ? 'Processing...' : (isEdit ? 'Update Banner' : 'Add Banner')} 
                                    disabled={isSubmitting}
                                />
                            </Col>
                        </Row>
                    </Col>
                </Row>
            </form>
        </fieldset>
    )
}

export default AddBanner