import { Col, Row } from 'react-bootstrap'
import { ButtonComp, InputComp } from '../../bootstrap'
import { useModal } from '../../../hooks/useModal'
import { useEffect, useState, useRef } from 'react'
import { useDispatch } from 'react-redux'
import { errorNotify, successNotify } from '../../../Toastify/Toastify'
import axios from 'axios'
import { BASE_URL } from '../../../config/service'
import { GetAchievements, GetHomePageBanners } from '../../../store/slices/propertyManagementSlice/propertyManagementSlice'

const AddAchievement = ({ isEdit, data, setIsEdit, onSuccess }) => {
    console.log('[data]', data)
    const { handleModalClose } = useModal()
    const dispatch = useDispatch();
    const fileInputRef = useRef();

    // Achievement form states
    const [cardTitle, setCardTitle] = useState("")
    const [cardContent, setCardContent] = useState("")
    const [imgAlt, setImgAlt] = useState("")
    const [imagePath, setImagePath] = useState(null)
    const [fileName, setFileName] = useState('No file chosen')
    const [isSubmitting, setIsSubmitting] = useState(false)

    useEffect(() => {
        if (isEdit && data) {
            setCardTitle(data.card_title || "")
            setCardContent(data.card_content || "")
            setImgAlt(data.img_alt || "")
            setFileName(data.image_path ? 'Current image' : 'No file chosen')
        }
    }, [isEdit, data])

    const handleFileChange = (event) => {
        const file = event.target.files[0];
        if (file) {
            setImagePath(file);
            setFileName(file.name);
        } else {
            setImagePath(null);
            setFileName('No file chosen');
        }
    };

    const handleButtonClick = () => {
        fileInputRef.current.click();
    };

    const addAchievementHandler = async (e) => {
        e.preventDefault()
        setIsSubmitting(true)

        // Validation
        if (!cardTitle || !cardContent || !imgAlt) {
            errorNotify("Required Fields are missing!")
            setIsSubmitting(false)
            return
        }

        try {
            const formData = new FormData();
            formData.append('card_title', cardTitle);
            formData.append('card_content', cardContent);
            formData.append('img_alt', imgAlt);

            if (imagePath) {
                formData.append('image_path', imagePath);
            }

            // Get auth token from localStorage
            const token = localStorage.getItem('token');
            if (!token) {
                errorNotify("Authentication token not found!")
                setIsSubmitting(false)
                return
            }

            const config = {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'multipart/form-data'
                }
            };

            let response;
            if (!isEdit) {
                // Create new achievement
                response = await axios.post(`${BASE_URL}/home/icon-box`, formData, config);
                successNotify("Achievement added successfully!");

            } else {
                // Update existing achievement
                response = await axios.patch(`${BASE_URL}/home/icon-box/${data.id}`, formData, config);
                successNotify("Achievement updated successfully!");
            }
            dispatch(GetAchievements())

            handleModalClose();
            setIsEdit(false);

            // Reset form
            setCardTitle("");
            setCardContent("");
            setImgAlt("");
            setImagePath(null);
            setFileName('No file chosen');

            // Call success callback if provided
            if (onSuccess) {
                onSuccess();
            }

        } catch (error) {
            console.error('Achievement submission error:', error);
            const errorMessage = error.response?.data?.message || error.message || "Something went wrong!";
            errorNotify(errorMessage);
        } finally {
            setIsSubmitting(false);
        }
    }

    return (
        <fieldset className="reset mt-3 p-4 pt-0 t-0 ">
            <legend className="reset mb-3 side-heading px-5 mx-auto">{isEdit ? 'Update' : 'Add'} Achievement</legend>
            <form onSubmit={addAchievementHandler} >
                <Row className='mt-3'>
                    <Col xxl={12} className="">
                        <Row>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Card Title</h5>
                                <div className='property-no'>
                                    <InputComp
                                        onChange={(e) => setCardTitle(e.target.value)}
                                        required={true}
                                        label={"Enter Card Title: *"}
                                        type={"text"}
                                        placeholder={"Card Title"}
                                        controlId={"floatingInput-1"}
                                        name="cardTitle"
                                        value={cardTitle}
                                    />
                                </div>
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Card Content</h5>
                                <div className='property-no'>
                                    <InputComp
                                        onChange={(e) => setCardContent(e.target.value)}
                                        required={true}
                                        label={"Enter Card Content: *"}
                                        type={"text"}
                                        placeholder={"Card Content"}
                                        controlId={"floatingInput-2"}
                                        name="cardContent"
                                        value={cardContent}
                                    />
                                </div>
                            </Col>
                        </Row>



                        <Row className='mt-3'>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Achievement Image</h5>
                                <div className='file-upload_comp'>
                                    <input
                                        type="file"
                                        ref={fileInputRef}
                                        onChange={handleFileChange}
                                        accept="image/*"
                                        style={{ display: 'none' }}
                                    />
                                    <button
                                        type="button"
                                        onClick={handleButtonClick}
                                        className="btn btn-outline-primary"
                                    >
                                        Choose file
                                    </button>
                                    <span className="ms-2">{fileName}</span>
                                </div>
                                {!isEdit && (
                                    <small className="text-danger">* Achievement image is required for new achievements</small>
                                )}
                            </Col>
                            <Col lg={6} className='bg-lightgray p-3'>
                                <h5 className='pb-3'>{isEdit ? 'Update' : 'Add'} Image Alt Text</h5>
                                <div className='property-no'>
                                    <InputComp
                                        onChange={(e) => setImgAlt(e.target.value)}
                                        required={true}
                                        label={"Enter Image Alt Text: *"}
                                        type={"text"}
                                        placeholder={"Image Alt Text"}
                                        controlId={"floatingInput-5"}
                                        name="imgAlt"
                                        value={imgAlt}
                                    />
                                </div>
                            </Col>
                        </Row>

                        <Row>
                            <Col lg={12} className=' mt-5'>
                                <ButtonComp
                                    type="submit"
                                    variant={"outline-primary"}
                                    className="btn-main-clr text-white"
                                    btnText={isSubmitting ? 'Processing...' : (isEdit ? 'Update Achievement' : 'Add Achievement')}
                                    disabled={isSubmitting}
                                />
                            </Col>
                        </Row>
                    </Col>
                </Row>
            </form>
        </fieldset>
    )
}

export default AddAchievement