import React, { useState, useEffect } from 'react';
import {
  Box,
  TextField,
  Button,
  Typography,
  Paper,
  Grid,
  Card,
  CardContent,
  IconButton,
  Divider,
  Container,
  Stack,
  CircularProgress,
  Alert,
  Switch,
  FormControlLabel
} from '@mui/material';
import { Add as AddIcon, Delete as DeleteIcon } from '@mui/icons-material';
import { useSelector, useDispatch } from 'react-redux';
import { GetAllServicesPages, SaveWebsiteDesignTemplatePage, UpdateWebsiteDesignTemplatePage, UploadServiceMediaAction } from '../../store/slices/servicesSlice/servicesSlice';
import { toast } from 'react-toastify';
import { GetAllServicesTabs } from '../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { useModal } from '../../hooks/useModal';

const WebsiteDesignServiceForm = ({ isEdit, initialData, onSuccess, handleModalClose, setIsEdit }) => {
  const { services } = useSelector(state => state.ServicesReducer);
  const dispatch = useDispatch();
  const [uploadingFiles, setUploadingFiles] = useState({});
  const [uploadErrors, setUploadErrors] = useState({});
  const { handleModalClose: defaultHandleModalClose } = useModal()

  // Use passed handleModalClose if available, otherwise use default
  const modalCloseHandler = handleModalClose || defaultHandleModalClose;

  console.log('[services]', services);

  const [formData, setFormData] = useState({
    page_heading: "",
    enable_review: false,
    enable_faq: false,
    banner: {
      banner_title_heading: "",
      banner_text: "",
      background_image_url: ""
    },
    services_tabs_section_image: "",
    service_section_Tabs: {
      section_heading: "",
      section_text: "",
      features: [
        {
          title: "",
          text: ""
        }
      ]
    },
    services_cards: [
      {
        title: "",
        description: "",
        image: ""
      }
    ],
    why_choose_us_section: {
      section_heading: "",
      section_text: "",
      our_qualities: [
        {
          title: "",
          description: ""
        }
      ]
    },
    info_cards_section: {
      section_heading: "",
      cards: [
        {
          title: "",
          services: [
            {
              image_url: "",
              title: "",
              text: ""
            }
          ]
        }
      ]
    }
  });

  // Pre-fill form data when in edit mode
  useEffect(() => {
    if (isEdit && initialData) {
      console.log('initialData', initialData);
      // Ensure all sections have proper structure even if they're null in the API response
      const processedData = {
        page_heading: initialData.page_heading || "",
        enable_review: initialData.enable_review || false,
        enable_faq: initialData.enable_faq || false,
        banner: {
          banner_title_heading: initialData.banner?.banner_title_heading || "",
          banner_text: initialData.banner?.banner_text || "",
          background_image_url: initialData.banner?.background_image_url || ""
        },
        services_tabs_section_image: initialData.services_tabs_section_image || "",
        service_section_Tabs: {
          section_heading: initialData.service_section_Tabs?.section_heading || "",
          section_text: initialData.service_section_Tabs?.section_text || "",
          features: initialData.service_section_Tabs?.features || [
            {
              title: "",
              text: ""
            }
          ]
        },
        services_cards: initialData.services_cards || [
          {
            title: "",
            description: "",
            image: ""
          }
        ],
        why_choose_us_section: {
          section_heading: initialData.why_choose_us_section?.section_heading || "",
          section_text: initialData.why_choose_us_section?.section_text || "",
          our_qualities: initialData.why_choose_us_section?.our_qualities || [
            {
              title: "",
              description: ""
            }
          ]
        },
        info_cards_section: {
          section_heading: initialData.info_cards_section?.section_heading || "",
          cards: initialData.info_cards_section?.cards || [
            {
              title: "",
              services: [
                {
                  image_url: "",
                  title: "",
                  text: ""
                }
              ]
            }
          ]
        }
      };
      setFormData(processedData);
    }
  }, [isEdit, initialData]);

  // Debug: Log formData changes
  useEffect(() => {
    console.log('formData changed:', formData);
  }, [formData]);

  const handleSubmit = (event) => {
    event.preventDefault();
    console.log('Form Data:', formData);
    if (formData.page_heading) {
      if (isEdit) {
        console.log('initialData', initialData);
        console.log('formData', formData);
        
        dispatch(UpdateWebsiteDesignTemplatePage({ id: initialData.id, data: formData }))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Updated');
              modalCloseHandler();
              dispatch(GetAllServicesPages());
              if (onSuccess) onSuccess(response.data);
              if (setIsEdit) setIsEdit(false);
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          });
      } else {
        dispatch(SaveWebsiteDesignTemplatePage(formData))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Created');
              modalCloseHandler();
              dispatch(GetAllServicesPages());
              if (onSuccess) onSuccess(response.data);
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          });
      }
    } else {
      toast.error('Page Heading is required');
    }
  };

  const handleInputChange = (section, subsection, index, field, subIndex = null) => (event) => {
    const value = event.target.value;
    
    if (section && !subsection && !index && !field && !subIndex) {
      // Handle direct field (like page_heading)
      setFormData(prev => ({
        ...prev,
        [section]: value
      }));
    } else if (section && subsection && index !== null && field && subIndex !== null) {
      // Handle deeply nested array item field (e.g., info_cards_section.cards[index].services[subIndex].title)
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? {
              ...item,
              services: item.services.map((service, j) =>
                j === subIndex ? { ...service, [field]: value } : service
              )
            } : item
          )
        }
      }));
    } else if (section && subsection && index !== null && field) {
      // Handle nested array item field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? { ...item, [field]: value } : item
          )
        }
      }));
    } else if (section && subsection) {
      // Handle nested object field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: value
        }
      }));
    }
  };

  const handleSwitchChange = (field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.checked
    }));
  };

  const handleFileChange = (section, subsection, index, field, subIndex = null) => async (event) => {
    const file = event.target.files[0];
    if (!file) return;

    // Create a unique key for this file upload
    const uploadKey = `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}_${subIndex !== null ? subIndex : 'none'}`;

    // Set uploading state
    setUploadingFiles(prev => ({ ...prev, [uploadKey]: true }));
    setUploadErrors(prev => ({ ...prev, [uploadKey]: null }));

    try {
      const result = await dispatch(UploadServiceMediaAction({ file })).unwrap();

      if (result.success && result.data?.file_url) {
        // Update form data with the returned file_url
        if (section && subsection && index !== null && field && subIndex !== null) {
          // Handle deeply nested array item field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? {
                  ...item,
                  services: item.services.map((service, j) =>
                    j === subIndex ? { ...service, [field]: result.data.file_url } : service
                  )
                } : item
              )
            }
          }));
        } else if (section && subsection && index !== null && field) {
          // Handle nested array item field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? { ...item, [field]: result.data.file_url } : item
              )
            }
          }));
        } else if (section && subsection) {
          // Handle nested object field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: result.data.file_url
            }
          }));
        } else if (section) {
          // Handle direct field
          setFormData(prev => ({
            ...prev,
            [section]: result.data.file_url
          }));
        }
      }
    } catch (error) {
      console.error('File upload failed:', error);
      setUploadErrors(prev => ({
        ...prev,
        [uploadKey]: error || 'File upload failed'
      }));
    } finally {
      setUploadingFiles(prev => ({ ...prev, [uploadKey]: false }));
    }
  };

  const getUploadKey = (section, subsection, index, field, subIndex = null) => {
    return `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}_${subIndex !== null ? subIndex : 'none'}`;
  };

  const isUploading = (section, subsection, index, field, subIndex = null) => {
    const key = getUploadKey(section, subsection, index, field, subIndex);
    return uploadingFiles[key] || false;
  };

  const getUploadError = (section, subsection, index, field, subIndex = null) => {
    const key = getUploadKey(section, subsection, index, field, subIndex);
    return uploadErrors[key] || null;
  };

  const addFeature = () => {
    setFormData(prev => ({
      ...prev,
      service_section_Tabs: {
        ...prev.service_section_Tabs,
        features: [...prev.service_section_Tabs.features, { title: "", text: "" }]
      }
    }));
  };

  const removeFeature = (index) => {
    setFormData(prev => ({
      ...prev,
      service_section_Tabs: {
        ...prev.service_section_Tabs,
        features: prev.service_section_Tabs.features.filter((_, i) => i !== index)
      }
    }));
  };

  const addServiceCard = () => {
    setFormData(prev => ({
      ...prev,
      services_cards: [...prev.services_cards, { title: "", description: "", image: "" }]
    }));
  };

  const removeServiceCard = (index) => {
    setFormData(prev => ({
      ...prev,
      services_cards: prev.services_cards.filter((_, i) => i !== index)
    }));
  };

  const addQuality = () => {
    setFormData(prev => ({
      ...prev,
      why_choose_us_section: {
        ...prev.why_choose_us_section,
        our_qualities: [...prev.why_choose_us_section.our_qualities, { title: "", description: "" }]
      }
    }));
  };

  const removeQuality = (index) => {
    setFormData(prev => ({
      ...prev,
      why_choose_us_section: {
        ...prev.why_choose_us_section,
        our_qualities: prev.why_choose_us_section.our_qualities.filter((_, i) => i !== index)
      }
    }));
  };

  const addInfoCard = () => {
    setFormData(prev => ({
      ...prev,
      info_cards_section: {
        ...prev.info_cards_section,
        cards: [...prev.info_cards_section.cards, { title: "", services: [{ image_url: "", title: "", text: "" }] }]
      }
    }));
  };

  const removeInfoCard = (index) => {
    setFormData(prev => ({
      ...prev,
      info_cards_section: {
        ...prev.info_cards_section,
        cards: prev.info_cards_section.cards.filter((_, i) => i !== index)
      }
    }));
  };

  const addServiceToCard = (cardIndex) => {
    setFormData(prev => ({
      ...prev,
      info_cards_section: {
        ...prev.info_cards_section,
        cards: prev.info_cards_section.cards.map((card, i) =>
          i === cardIndex ? {
            ...card,
            services: [...card.services, { image_url: "", title: "", text: "" }]
          } : card
        )
      }
    }));
  };

  const removeServiceFromCard = (cardIndex, serviceIndex) => {
    setFormData(prev => ({
      ...prev,
      info_cards_section: {
        ...prev.info_cards_section,
        cards: prev.info_cards_section.cards.map((card, i) =>
          i === cardIndex ? {
            ...card,
            services: card.services.filter((_, j) => j !== serviceIndex)
          } : card
        )
      }
    }));
  };



  return (
    <Container maxWidth="lg" sx={{ py: 4 }}>
      {/* Safety check - ensure formData is properly initialized */}
      {!formData || !formData.service_section_Tabs ? (
        <Box sx={{ display: 'flex', justifyContent: 'center', py: 4 }}>
          <CircularProgress />
        </Box>
      ) : (
        <Box component="form" onSubmit={handleSubmit}>
        {/* Page Heading */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Page Heading
            </Typography>
            <TextField
              fullWidth
              label="Page Heading"
              value={formData.page_heading}
              onChange={(e) => handleInputChange('page_heading', null, null, null)(e)}
              margin="normal"
            />
          </CardContent>
        </Card>

        {/* Banner Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Banner Section
            </Typography>
            <Grid container spacing={2}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Banner Title Heading"
                  value={formData.banner?.banner_title_heading || ""}
                  onChange={handleInputChange('banner', 'banner_title_heading', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Banner Text"
                  value={formData.banner?.banner_text || ""}
                  onChange={handleInputChange('banner', 'banner_text', null, null)}
                  margin="normal"
                  multiline
                  rows={3}
                />
              </Grid>
              <Grid item xs={12}>
                <Typography variant="subtitle2" gutterBottom>
                  Background Image
                </Typography>
                <input
                  type="file"
                  accept="image/*"
                  onChange={(e) => handleFileChange('banner', 'background_image_url', null, null)(e)}
                  style={{ width: '100%' }}
                />
                {isUploading('banner', 'background_image_url', null, null) && (
                  <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                    <CircularProgress size={20} />
                    <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                  </Box>
                )}
                {getUploadError('banner', 'background_image_url', null, null) && (
                  <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('banner', 'background_image_url', null, null)}</Alert>
                )}
                {formData.banner?.background_image_url && (
                  <Alert severity="success" sx={{ mt: 1 }}>
                    <Typography variant="body2">
                      <strong>Uploaded:</strong> {formData.banner.background_image_url}
                    </Typography>
                  </Alert>
                )}
              </Grid>
            </Grid>
          </CardContent>
        </Card>

        {/* Services Tabs Section Image */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Services Tabs Section Image
            </Typography>
            <input
              type="file"
              accept="image/*"
              onChange={(e) => handleFileChange('services_tabs_section_image', null, null, null)(e)}
              style={{ width: '100%' }}
            />
            {isUploading('services_tabs_section_image', null, null, null) && (
              <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                <CircularProgress size={20} />
                <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
              </Box>
            )}
            {getUploadError('services_tabs_section_image', null, null, null) && (
              <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('services_tabs_section_image', null, null, null)}</Alert>
            )}
            {formData.services_tabs_section_image && (
              <Alert severity="success" sx={{ mt: 1 }}>
                <Typography variant="body2">
                  <strong>Uploaded:</strong> {formData.services_tabs_section_image}
                </Typography>
              </Alert>
            )}
          </CardContent>
        </Card>

        {/* Service Section Tabs */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Service Section Tabs
            </Typography>
            <Grid container spacing={2} sx={{ mb: 3 }}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Heading"
                  value={formData.service_section_Tabs?.section_heading || ""}
                  onChange={handleInputChange('service_section_Tabs', 'section_heading', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Text"
                  value={formData.service_section_Tabs?.section_text || ""}
                  onChange={handleInputChange('service_section_Tabs', 'section_text', null, null)}
                  margin="normal"
                  multiline
                  rows={3}
                />
              </Grid>
            </Grid>
            <Stack spacing={3}>
              {formData.service_section_Tabs?.features?.map((feature, index) => (
                <Paper key={index} sx={{ p: 2 }}>
                  <Grid container spacing={2}>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Feature Title"
                        value={feature.title}
                        onChange={handleInputChange('service_section_Tabs', 'features', index, 'title')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Feature Text"
                        value={feature.text}
                        onChange={handleInputChange('service_section_Tabs', 'features', index, 'text')}
                        margin="normal"
                        multiline
                        rows={2}
                      />
                    </Grid>
                    <Grid item xs={12} style={{ display: 'flex', justifyContent: 'flex-end' }}>
                      <IconButton
                        onClick={() => removeFeature(index)}
                        color="error"
                        disabled={formData.service_section_Tabs?.features?.length === 1}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </Grid>
                  </Grid>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addFeature}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Feature
            </Button>
          </CardContent>
        </Card>

        {/* Services Cards */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Services Cards
            </Typography>
            <Stack spacing={3}>
              {formData.services_cards?.map((card, index) => (
                <Paper key={index} sx={{ p: 2 }}>
                  <Grid container spacing={2}>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Card Title"
                        value={card.title}
                        onChange={handleInputChange('services_cards', null, index, 'title')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Description"
                        value={card.description}
                        onChange={handleInputChange('services_cards', null, index, 'description')}
                        margin="normal"
                        multiline
                        rows={3}
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <Typography variant="subtitle2" gutterBottom>
                        Card Image
                      </Typography>
                      <input
                        type="file"
                        accept="image/*"
                        onChange={handleFileChange('services_cards', null, index, 'image')}
                        style={{ width: '100%' }}
                      />
                      {isUploading('services_cards', null, index, 'image') && (
                        <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                          <CircularProgress size={20} />
                          <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                        </Box>
                      )}
                      {getUploadError('services_cards', null, index, 'image') && (
                        <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('services_cards', null, index, 'image')}</Alert>
                      )}
                      {card.image && (
                        <Alert severity="success" sx={{ mt: 1 }}>
                          <Typography variant="body2">
                            <strong>Uploaded:</strong> {card.image}
                          </Typography>
                        </Alert>
                      )}
                    </Grid>
                    <Grid item xs={12} style={{ display: 'flex', justifyContent: 'flex-end' }}>
                      <IconButton
                        onClick={() => removeServiceCard(index)}
                        color="error"
                        disabled={formData.services_cards?.length === 1}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </Grid>
                  </Grid>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addServiceCard}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Service Card
            </Button>
          </CardContent>
        </Card>

        {/* Why Choose Us Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Why Choose Us Section
            </Typography>
            <Grid container spacing={2} sx={{ mb: 3 }}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Heading"
                  value={formData.why_choose_us_section?.section_heading || ""}
                  onChange={handleInputChange('why_choose_us_section', 'section_heading', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Text"
                  value={formData.why_choose_us_section?.section_text || ""}
                  onChange={handleInputChange('why_choose_us_section', 'section_text', null, null)}
                  margin="normal"
                  multiline
                  rows={3}
                />
              </Grid>
            </Grid>
            <Stack spacing={3}>
              {formData.why_choose_us_section?.our_qualities?.map((quality, index) => (
                <Paper key={index} sx={{ p: 2 }}>
                  <Grid container spacing={2}>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Quality Title"
                        value={quality.title}
                        onChange={handleInputChange('why_choose_us_section', 'our_qualities', index, 'title')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Quality Description"
                        value={quality.description}
                        onChange={handleInputChange('why_choose_us_section', 'our_qualities', index, 'description')}
                        margin="normal"
                        multiline
                        rows={4}
                      />
                    </Grid>
                    <Grid item xs={12} style={{ display: 'flex', justifyContent: 'flex-end' }}>
                      <IconButton
                        onClick={() => removeQuality(index)}
                        color="error"
                        disabled={formData.why_choose_us_section?.our_qualities?.length === 1}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </Grid>
                  </Grid>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addQuality}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Quality
            </Button>
          </CardContent>
        </Card>

        {/* Info Cards Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Info Cards Section
            </Typography>
            <TextField
              fullWidth
              label="Section Heading"
              value={formData.info_cards_section?.section_heading || ""}
              onChange={handleInputChange('info_cards_section', 'section_heading', null, null)}
              margin="normal"
              style={{ marginBottom: '16px' }}
            />
            <Stack spacing={4}>
              {formData.info_cards_section?.cards?.map((card, cardIndex) => (
                <Paper key={cardIndex} sx={{ p: 3, border: '1px solid #e0e0e0' }}>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
                    <Typography variant="h6" gutterBottom>
                      Card {cardIndex + 1}
                    </Typography>
                    <IconButton
                      onClick={() => removeInfoCard(cardIndex)}
                      color="error"
                      disabled={formData.info_cards_section?.cards?.length === 1}
                    >
                      <DeleteIcon />
                    </IconButton>
                  </Box>
                  <TextField
                    fullWidth
                    label="Card Title"
                    value={card.title}
                    onChange={handleInputChange('info_cards_section', 'cards', cardIndex, 'title')}
                    margin="normal"
                    style={{ marginBottom: '16px' }}
                  />
                  <Divider sx={{ my: 3 }} />
                  <Typography variant="subtitle1" gutterBottom>
                    Services
                  </Typography>
                  <Stack spacing={2}>
                    {card.services?.map((service, serviceIndex) => (
                      <Paper key={serviceIndex} sx={{ p: 2, bgcolor: '#f9f9f9' }}>
                        <Grid container spacing={2}>

                          <Grid item xs={12} md={12}>
                            <TextField
                              fullWidth
                              label="Service Title"
                              value={service.title}
                              onChange={handleInputChange('info_cards_section', 'cards', cardIndex, 'title', serviceIndex)}
                              margin="normal"
                            />
                          </Grid>
                          <Grid item xs={12} md={12}>
                            <TextField
                              fullWidth
                              label="Service Text"
                              value={service.text}
                              onChange={handleInputChange('info_cards_section', 'cards', cardIndex, 'text', serviceIndex)}
                              margin="normal"
                              multiline
                              rows={2}
                            />
                          </Grid>
                          <Grid item xs={12} md={4}>
                            <Typography variant="subtitle2" gutterBottom>
                              Service Image
                            </Typography>
                            <input
                              type="file"
                              accept="image/*"
                              onChange={handleFileChange('info_cards_section', 'cards', cardIndex, 'image_url', serviceIndex)}
                              style={{ width: '100%' }}
                            />
                            {isUploading('info_cards_section', 'cards', cardIndex, 'image_url', serviceIndex) && (
                              <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                                <CircularProgress size={20} />
                                <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                              </Box>
                            )}
                            {getUploadError('info_cards_section', 'cards', cardIndex, 'image_url', serviceIndex) && (
                              <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('info_cards_section', 'cards', cardIndex, 'image_url', serviceIndex)}</Alert>
                            )}
                            {service.image_url && (
                              <Alert severity="success" sx={{ mt: 1 }}>
                                <Typography variant="body2">
                                  <strong>Uploaded:</strong> {service.image_url}
                                </Typography>
                              </Alert>
                            )}
                          </Grid>
                          <Grid item xs={12} style={{ display: 'flex', justifyContent: 'flex-end' }}>
                            <IconButton
                              onClick={() => removeServiceFromCard(cardIndex, serviceIndex)}
                              color="error"
                              disabled={card.services?.length === 1}
                            >
                              <DeleteIcon />
                            </IconButton>
                          </Grid>
                        </Grid>
                      </Paper>
                    ))}
                  </Stack>
                  <Button
                    startIcon={<AddIcon />}
                    onClick={() => addServiceToCard(cardIndex)}
                    variant="outlined"
                    sx={{ mt: 2 }}
                  >
                    Add Service
                  </Button>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addInfoCard}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Info Card
            </Button>
          </CardContent>
        </Card>

        {/* Page Settings Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Page Settings
            </Typography>
            <Box sx={{ display: 'flex', gap: 4, flexWrap: 'wrap' }}>
              <FormControlLabel
                control={
                  <Switch
                    checked={formData.enable_review}
                    onChange={handleSwitchChange('enable_review')}
                    color="primary"
                  />
                }
                label="Enable Review on this page"
              />
              <FormControlLabel
                control={
                  <Switch
                    checked={formData.enable_faq}
                    onChange={handleSwitchChange('enable_faq')}
                    color="primary"
                  />
                }
                label="Enable FAQ on this page"
              />
            </Box>
          </CardContent>
        </Card>

        {/* Submit Button */}
        <Box sx={{ display: 'flex', justifyContent: 'center', mt: 4 }}>
          <Button
            type="submit"
            variant="contained"
            size="large"
            sx={{ minWidth: 200 }}
          >
            {isEdit ? 'Update Page' : 'Submit Form'}
          </Button>
        </Box>
      </Box>
      )}
    </Container>
  );
};

export default WebsiteDesignServiceForm;