import React, { useState, useEffect } from 'react';
import {
  Box,
  TextField,
  Button,
  Typography,
  Paper,
  Grid,
  Card,
  CardContent,
  IconButton,
  Container,
  Stack,
  CircularProgress,
  Alert,
  Switch,
  FormControlLabel
} from '@mui/material';
import { Add as AddIcon, Delete as DeleteIcon } from '@mui/icons-material';
import { useSelector, useDispatch } from 'react-redux';
import { GetAllServicesPages, SocialMediaMarketingTemplatePage, UploadServiceMediaAction, UpdateSocialMediaMarkettingTemplatePage } from '../../store/slices/servicesSlice/servicesSlice';
import { toast } from 'react-toastify';
import { GetAllServicesTabs } from '../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { useModal } from '../../hooks/useModal';

const SocialMediaMarketingServiceForm = ({ isEdit, initialData, onSuccess, handleModalClose, setIsEdit }) => {
  const { handleModalClose: defaultHandleModalClose } = useModal()
  const { services } = useSelector(state => state.ServicesReducer);
  const dispatch = useDispatch();
  const [uploadingFiles, setUploadingFiles] = useState({});
  const [uploadErrors, setUploadErrors] = useState({});

  // Use passed handleModalClose if available, otherwise use default
  const modalCloseHandler = handleModalClose || defaultHandleModalClose;

  console.log('[services]', services);

  const [formData, setFormData] = useState({
    page_heading: "",
    enable_review: false,
    enable_faq: false,
    banner: {
      banner_title_heading: "",
      banner_span: "",
      background_image_url: ""
    },
    stand_out_section: {
      image: [""]
    },
    our_achievements_section: {
      section_headign: "",
      section_text: "",
      features: [
        {
          icon_url: "",
          title: "",
          text: ""
        }
      ]
    },
    our_services_section: {
      section_heading: "",
      section_span: "",
      section_text: "",
      services_cards: {
        section_headign: "",
        section_cards: [
          {
            icon_url: "",
            title: "",
            description: ""
          }
        ]
      }
    },
    our_experties: {
      section_heading: "",
      section_span: "",
      section_text: "",
      sections_cards: {
        sections_heading: "",
        sections_cards: [
          {
            icon_url: "",
            title: "",
            description: ""
          }
        ]
      }
    },
    technologies_we_work_section: {
      section_heading: "",
      section_background: "",
      image: [""]
    },
    our_profile: {
      background_image_url: "",
      image: "",
      section_heading: "",
      button_text: "",
      button_url: "",
      button_span: ""
    },
    about_us: {
      section_heading: "",
      section_span: "",
      section_text: "",
      button_text: "",
      button_url: "",
      section_background: ""
    }
  });

  // Pre-fill form data when in edit mode
  useEffect(() => {
    if (isEdit && initialData) {
      // Ensure all sections have proper structure even if they're null in the API response
      const processedData = {
        ...initialData,
        banner: initialData.banner || {
          banner_title_heading: "",
          banner_span: "",
          background_image_url: ""
        },
        stand_out_section: initialData.stand_out_section || {
          image: [""]
        },
        our_achievements_section: initialData.our_achievements_section || {
          section_headign: "",
          section_text: "",
          features: [
            {
              icon_url: "",
              title: "",
              text: ""
            }
          ]
        },
        our_services_section: initialData.our_services_section || {
          section_heading: "",
          section_span: "",
          section_text: "",
          services_cards: {
            section_headign: "",
            section_cards: [
              {
                icon_url: "",
                title: "",
                description: ""
              }
            ]
          }
        },
        our_experties: initialData.our_experties || {
          section_heading: "",
          section_span: "",
          section_text: "",
          sections_cards: {
            sections_heading: "",
            sections_cards: [
              {
                icon_url: "",
                title: "",
                description: ""
              }
            ]
          }
        },
        technologies_we_work_section: initialData.technologies_we_work_section || {
          section_heading: "",
          section_background: "",
          image: [""]
        },
        our_profile: initialData.our_profile || {
          background_image_url: "",
          image: "",
          section_heading: "",
          button_text: "",
          button_url: "",
          button_span: ""
        },
        about_us: initialData.about_us || {
          section_heading: "",
          section_span: "",
          section_text: "",
          button_text: "",
          button_url: "",
          section_background: ""
        }
      };
      setFormData(processedData);
    }
  }, [isEdit, initialData]);

  const handleSubmit = (event) => {
    event.preventDefault();
    console.log('Form Data:', formData);
    if (formData.page_heading) {
      if (isEdit) {
        console.log('initialData', initialData)
        console.log('formData', formData)

        dispatch(UpdateSocialMediaMarkettingTemplatePage({ id: initialData.id, data: formData }))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Updated');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      } else {
        dispatch(SocialMediaMarketingTemplatePage(formData))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Created');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      }
    } else {
      toast.error('Page Heading is required');
    }
  };

  const handleInputChange = (section, subsection, index, field) => {
    if (section && subsection && index !== null && field) {
      // Handle nested array item field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? { ...item, [field]: event.target.value } : item
          )
        }
      }));
    } else if (section && subsection && index !== null) {
      // Handle array item (for services array)
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? event.target.value : item
          )
        }
      }));
    } else if (section && subsection) {
      // Handle nested object field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: event.target.value
        }
      }));
    } else if (section) {
      // Handle direct field
      setFormData(prev => ({
        ...prev,
        [section]: event.target.value
      }));
    }
  };

  const handleSwitchChange = (field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.checked
    }));
  };

  const handleFileChange = async (section, subsection, index, field) => {
    const file = event.target.files[0];
    if (!file) return;

    // Create a unique key for this file upload
    const uploadKey = `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
    
    // Set uploading state
    setUploadingFiles(prev => ({ ...prev, [uploadKey]: true }));
    setUploadErrors(prev => ({ ...prev, [uploadKey]: null }));

    try {
      const result = await dispatch(UploadServiceMediaAction({ file })).unwrap();
      
      if (result.success && result.data?.file_url) {
        // Update form data with the returned file_url
        if (section && subsection && index !== null && field) {
          // Handle nested array item field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? { ...item, [field]: result.data.file_url } : item
              )
            }
          }));
        } else if (section && subsection) {
          // Handle nested object field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: result.data.file_url
            }
          }));
        } else if (section) {
          // Handle direct field
          setFormData(prev => ({
            ...prev,
            [section]: result.data.file_url
          }));
        }
      }
    } catch (error) {
      console.error('File upload failed:', error);
      setUploadErrors(prev => ({ 
        ...prev, 
        [uploadKey]: error || 'File upload failed' 
      }));
    } finally {
      setUploadingFiles(prev => ({ ...prev, [uploadKey]: false }));
    }
  };

  const getUploadKey = (section, subsection, index, field) => {
    return `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
  };

  const isUploading = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadingFiles[key] || false;
  };

  const getUploadError = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadErrors[key] || null;
  };

  const addStandOutImage = () => {
    setFormData(prev => ({
      ...prev,
      stand_out_section: {
        ...prev.stand_out_section,
        image: [...prev.stand_out_section.image, ""]
      }
    }));
  };

  const removeStandOutImage = (index) => {
    setFormData(prev => ({
      ...prev,
      stand_out_section: {
        ...prev.stand_out_section,
        image: prev.stand_out_section.image.filter((_, i) => i !== index)
      }
    }));
  };

  const addAchievementFeature = () => {
    setFormData(prev => ({
      ...prev,
      our_achievements_section: {
        ...prev.our_achievements_section,
        features: [...prev.our_achievements_section.features, { icon_url: "", title: "", text: "" }]
      }
    }));
  };

  const removeAchievementFeature = (index) => {
    setFormData(prev => ({
      ...prev,
      our_achievements_section: {
        ...prev.our_achievements_section,
        features: prev.our_achievements_section.features.filter((_, i) => i !== index)
      }
    }));
  };

  const addServiceCard = () => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services_cards: {
          ...prev.our_services_section.services_cards,
          section_cards: [...prev.our_services_section.services_cards.section_cards, { icon_url: "", title: "", description: "" }]
        }
      }
    }));
  };

  const removeServiceCard = (index) => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services_cards: {
          ...prev.our_services_section.services_cards,
          section_cards: prev.our_services_section.services_cards.section_cards.filter((_, i) => i !== index)
        }
      }
    }));
  };

  const addExpertiseCard = () => {
    setFormData(prev => ({
      ...prev,
      our_experties: {
        ...prev.our_experties,
        sections_cards: {
          ...prev.our_experties.sections_cards,
          sections_cards: [...prev.our_experties.sections_cards.sections_cards, { icon_url: "", title: "", description: "" }]
        }
      }
    }));
  };

  const removeExpertiseCard = (index) => {
    setFormData(prev => ({
      ...prev,
      our_experties: {
        ...prev.our_experties,
        sections_cards: {
          ...prev.our_experties.sections_cards,
          sections_cards: prev.our_experties.sections_cards.sections_cards.filter((_, i) => i !== index)
        }
      }
    }));
  };

  const addTechnologyImage = () => {
    setFormData(prev => ({
      ...prev,
      technologies_we_work_section: {
        ...prev.technologies_we_work_section,
        image: [...prev.technologies_we_work_section.image, ""]
      }
    }));
  };

  const removeTechnologyImage = (index) => {
    setFormData(prev => ({
      ...prev,
      technologies_we_work_section: {
        ...prev.technologies_we_work_section,
        image: prev.technologies_we_work_section.image.filter((_, i) => i !== index)
      }
    }));
  };



  return (
    <Container maxWidth="xl" sx={{ py: 4 }}>
      <Paper elevation={3} sx={{ p: 4 }}>

        <Box component="form" onSubmit={handleSubmit}>
          {/* Page Heading */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Heading
              </Typography>
              <TextField
                fullWidth
                label="Page Heading"
                value={formData.page_heading}
                onChange={(e) => handleInputChange('page_heading', null, null, null)}
                placeholder="Enter page heading"
                variant="outlined"
                margin="normal"
              />
            </CardContent>
          </Card>

          {/* Banner Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Banner Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Title Heading"
                    value={formData.banner.banner_title_heading}
                    onChange={(e) => handleInputChange('banner', 'banner_title_heading', null, null)}
                    placeholder="Enter banner title heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Span"
                    value={formData.banner.banner_span}
                    onChange={(e) => handleInputChange('banner', 'banner_span', null, null)}
                    placeholder="Enter banner span"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('banner', 'background_image_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('banner', 'background_image_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('banner', 'background_image_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('banner', 'background_image_url', null, null)}</Alert>
                  )}
                  {formData.banner.background_image_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.banner.background_image_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Stand Out Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Stand Out Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addStandOutImage}
                  
                >
                  Add Image
                </Button>
              </Box>
              <Stack spacing={2}>
                {formData.stand_out_section.image.map((image, index) => (
                  <Box key={index} display="flex" gap={2} alignItems="center">
                    <Typography variant="subtitle2" gutterBottom>
                      Image {index + 1}
                    </Typography>
                    <input
                      type="file"
                      accept="image/*"
                      onChange={(e) => handleFileChange('stand_out_section', 'image', index, null)}
                      style={{ flex: 1 }}
                    />
                    {isUploading('stand_out_section', 'image', index, null) && (
                      <Box sx={{ display: 'flex', alignItems: 'center' }}>
                        <CircularProgress size={20} />
                        <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                      </Box>
                    )}
                    {getUploadError('stand_out_section', 'image', index, null) && (
                      <Alert severity="error" sx={{ ml: 1 }}>{getUploadError('stand_out_section', 'image', index, null)}</Alert>
                    )}
                    {image && (
                      <Alert severity="success" sx={{ ml: 1 }}>
                        <Typography variant="body2">
                          <strong>Uploaded:</strong> {image}
                        </Typography>
                      </Alert>
                    )}
                    {formData.stand_out_section.image.length > 1 && (
                      <IconButton
                        color="error"
                        onClick={() => removeStandOutImage(index)}
                        size="small"
                      >
                        <DeleteIcon />
                      </IconButton>
                    )}
                  </Box>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Our Achievements Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Our Achievements Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addAchievementFeature}
                  
                >
                  Add Achievement
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.our_achievements_section.section_headign}
                onChange={(e) => handleInputChange('our_achievements_section', 'section_headign', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Text"
                value={formData.our_achievements_section.section_text}
                onChange={(e) => handleInputChange('our_achievements_section', 'section_text', null, null)}
                placeholder="Enter section text"
                variant="outlined"
                multiline
                rows={3}
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.our_achievements_section.features.map((feature, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Achievement {index + 1}</Typography>
                      {formData.our_achievements_section.features.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeAchievementFeature(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_achievements_section', 'features', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_achievements_section', 'features', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_achievements_section', 'features', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_achievements_section', 'features', index, 'icon_url')}</Alert>
                        )}
                        {feature.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {feature.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={feature.title}
                          onChange={(e) => handleInputChange('our_achievements_section', 'features', index, 'title')}
                          placeholder="Enter achievement title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Text"
                          value={feature.text}
                          onChange={(e) => handleInputChange('our_achievements_section', 'features', index, 'text')}
                          placeholder="Enter achievement text"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Our Services Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Our Services Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addServiceCard}
                  
                >
                  Add Service Card
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.our_services_section.section_heading}
                onChange={(e) => handleInputChange('our_services_section', 'section_heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Span"
                value={formData.our_services_section.section_span}
                onChange={(e) => handleInputChange('our_services_section', 'section_span', null, null)}
                placeholder="Enter section span"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Text"
                value={formData.our_services_section.section_text}
                onChange={(e) => handleInputChange('our_services_section', 'section_text', null, null)}
                placeholder="Enter section text"
                variant="outlined"
                multiline
                rows={3}
                margin="normal"
                sx={{ mb: 3 }}
              />
              <TextField
                fullWidth
                label="Services Cards Section Heading"
                value={formData.our_services_section.services_cards.section_headign}
                onChange={(e) => handleInputChange('our_services_section', 'services_cards', null, 'section_headign')}
                placeholder="Enter services cards section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.our_services_section.services_cards.section_cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Service Card {index + 1}</Typography>
                      {formData.our_services_section.services_cards.section_cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeServiceCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_services_section', 'services_cards', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_services_section', 'services_cards', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_services_section', 'services_cards', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_services_section', 'services_cards', index, 'icon_url')}</Alert>
                        )}
                        {card.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={card.title}
                          onChange={(e) => handleInputChange('our_services_section', 'services_cards', index, 'title')}
                          placeholder="Enter service title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={card.description}
                          onChange={(e) => handleInputChange('our_services_section', 'services_cards', index, 'description')}
                          placeholder="Enter service description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Our Expertise Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Our Expertise
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addExpertiseCard}
                  
                >
                  Add Expertise Card
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.our_experties.section_heading}
                onChange={(e) => handleInputChange('our_experties', 'section_heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Span"
                value={formData.our_experties.section_span}
                onChange={(e) => handleInputChange('our_experties', 'section_span', null, null)}
                placeholder="Enter section span"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Text"
                value={formData.our_experties.section_text}
                onChange={(e) => handleInputChange('our_experties', 'section_text', null, null)}
                placeholder="Enter section text"
                variant="outlined"
                multiline
                rows={3}
                margin="normal"
                sx={{ mb: 3 }}
              />
              <TextField
                fullWidth
                label="Sections Cards Heading"
                value={formData.our_experties.sections_cards.sections_heading}
                onChange={(e) => handleInputChange('our_experties', 'sections_cards', null, 'sections_heading')}
                placeholder="Enter sections cards heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.our_experties.sections_cards.sections_cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Expertise Card {index + 1}</Typography>
                      {formData.our_experties.sections_cards.sections_cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeExpertiseCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_experties', 'sections_cards', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_experties', 'sections_cards', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_experties', 'sections_cards', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_experties', 'sections_cards', index, 'icon_url')}</Alert>
                        )}
                        {card.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={card.title}
                          onChange={(e) => handleInputChange('our_experties', 'sections_cards', index, 'title')}
                          placeholder="Enter expertise title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={card.description}
                          onChange={(e) => handleInputChange('our_experties', 'sections_cards', index, 'description')}
                          placeholder="Enter expertise description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Technologies We Work Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Technologies We Work Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addTechnologyImage}
                  
                >
                  Add Image
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.technologies_we_work_section.section_heading}
                onChange={(e) => handleInputChange('technologies_we_work_section', 'section_heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Typography variant="subtitle1" gutterBottom sx={{ mb: 2 }}>
                Section Background
              </Typography>
              <input
                type="file"
                accept="image/*"
                onChange={(e) => handleFileChange('technologies_we_work_section', 'section_background', null, null)}
                style={{ width: '100%', marginBottom: '16px' }}
              />
              {isUploading('technologies_we_work_section', 'section_background', null, null) && (
                <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                  <CircularProgress size={20} />
                  <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                </Box>
              )}
              {getUploadError('technologies_we_work_section', 'section_background', null, null) && (
                <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('technologies_we_work_section', 'section_background', null, null)}</Alert>
              )}
              {formData.technologies_we_work_section.section_background && (
                <Alert severity="success" sx={{ mt: 1 }}>
                  <Typography variant="body2">
                    <strong>Uploaded:</strong> {formData.technologies_we_work_section.section_background}
                  </Typography>
                </Alert>
              )}
              <Stack spacing={2}>
                {formData.technologies_we_work_section.image.map((image, index) => (
                  <Box key={index} display="flex" gap={2} alignItems="center">
                    <Typography variant="subtitle2" gutterBottom>
                      Image {index + 1}
                    </Typography>
                    <input
                      type="file"
                      accept="image/*"
                      onChange={(e) => handleFileChange('technologies_we_work_section', 'image', index, null)}
                      style={{ flex: 1 }}
                    />
                    {isUploading('technologies_we_work_section', 'image', index, null) && (
                      <Box sx={{ display: 'flex', alignItems: 'center' }}>
                        <CircularProgress size={20} />
                        <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                      </Box>
                    )}
                    {getUploadError('technologies_we_work_section', 'image', index, null) && (
                      <Alert severity="error" sx={{ ml: 1 }}>{getUploadError('technologies_we_work_section', 'image', index, null)}</Alert>
                    )}
                    {image && (
                      <Alert severity="success" sx={{ ml: 1 }}>
                        <Typography variant="body2">
                          <strong>Uploaded:</strong> {image}
                        </Typography>
                      </Alert>
                    )}
                    {formData.technologies_we_work_section.image.length > 1 && (
                      <IconButton
                        color="error"
                        onClick={() => removeTechnologyImage(index)}
                        size="small"
                      >
                        <DeleteIcon />
                      </IconButton>
                    )}
                  </Box>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Our Profile Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Our Profile Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={6}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('our_profile', 'background_image_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('our_profile', 'background_image_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('our_profile', 'background_image_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_profile', 'background_image_url', null, null)}</Alert>
                  )}
                  {formData.our_profile.background_image_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.our_profile.background_image_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
                <Grid item xs={12} md={6}>
                  <Typography variant="subtitle1" gutterBottom>
                    Profile Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('our_profile', 'image', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('our_profile', 'image', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('our_profile', 'image', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_profile', 'image', null, null)}</Alert>
                  )}
                  {formData.our_profile.image && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.our_profile.image}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.our_profile.section_heading}
                    onChange={(e) => handleInputChange('our_profile', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={4}>
                  <TextField
                    fullWidth
                    label="Button Text"
                    value={formData.our_profile.button_text}
                    onChange={(e) => handleInputChange('our_profile', 'button_text', null, null)}
                    placeholder="Enter button text"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={4}>
                  <TextField
                    fullWidth
                    label="Button URL"
                    value={formData.our_profile.button_url}
                    onChange={(e) => handleInputChange('our_profile', 'button_url', null, null)}
                    placeholder="Enter button URL"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={4}>
                  <TextField
                    fullWidth
                    label="Button Span"
                    value={formData.our_profile.button_span}
                    onChange={(e) => handleInputChange('our_profile', 'button_span', null, null)}
                    placeholder="Enter button span"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* About Us Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                About Us Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.about_us.section_heading}
                    onChange={(e) => handleInputChange('about_us', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Span"
                    value={formData.about_us.section_span}
                    onChange={(e) => handleInputChange('about_us', 'section_span', null, null)}
                    placeholder="Enter section span"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.about_us.section_text}
                    onChange={(e) => handleInputChange('about_us', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={6}>
                  <TextField
                    fullWidth
                    label="Button Text"
                    value={formData.about_us.button_text}
                    onChange={(e) => handleInputChange('about_us', 'button_text', null, null)}
                    placeholder="Enter button text"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={6}>
                  <TextField
                    fullWidth
                    label="Button URL"
                    value={formData.about_us.button_url}
                    onChange={(e) => handleInputChange('about_us', 'button_url', null, null)}
                    placeholder="Enter button URL"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Section Background
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('about_us', 'section_background', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('about_us', 'section_background', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('about_us', 'section_background', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('about_us', 'section_background', null, null)}</Alert>
                  )}
                  {formData.about_us.section_background && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.about_us.section_background}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Page Settings Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Settings
              </Typography>
              <Box sx={{ display: 'flex', gap: 4, flexWrap: 'wrap' }}>
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_review}
                      onChange={handleSwitchChange('enable_review')}
                      color="primary"
                    />
                  }
                  label="Enable Review on this page"
                />
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_faq}
                      onChange={handleSwitchChange('enable_faq')}
                      color="primary"
                    />
                  }
                  label="Enable FAQ on this page"
                />
              </Box>
            </CardContent>
          </Card>

          {/* Submit Button */}
          <Box display="flex" justifyContent="center" mt={4}>
            <Button
              type="submit"
              variant="contained"
              size="large"
              
              sx={{ py: 1, px: 4, fontSize: '1rem' }}
            >
              {isEdit ? 'Update Page' : 'Submit Form'}
            </Button>
          </Box>
        </Box>
      </Paper>
    </Container>
  );
};

export default SocialMediaMarketingServiceForm;