import React, { useState, useEffect } from 'react';
import {
  Box,
  TextField,
  Button,
  Typography,
  Paper,
  Grid,
  Card,
  CardContent,
  IconButton,
  Container,
  Stack,
  CircularProgress,
  Alert,
  Switch,
  FormControlLabel
} from '@mui/material';
import { Add as AddIcon, Delete as DeleteIcon } from '@mui/icons-material';
import { useSelector, useDispatch } from 'react-redux';
import { GetAllServicesPages, SeoTemplatePage, UploadServiceMediaAction, UpdateSeoServiceTemplatePage } from '../../store/slices/servicesSlice/servicesSlice';
import { toast } from 'react-toastify';
import { GetAllServicesTabs } from '../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { useModal } from '../../hooks/useModal';

const SeoServiceForm = ({ isEdit, initialData, onSuccess, handleModalClose, setIsEdit }) => {
  const { handleModalClose: defaultHandleModalClose } = useModal()
  const { services } = useSelector(state => state.ServicesReducer);
  const dispatch = useDispatch();
  const [uploadingFiles, setUploadingFiles] = useState({});
  const [uploadErrors, setUploadErrors] = useState({});

  // Use passed handleModalClose if available, otherwise use default
  const modalCloseHandler = handleModalClose || defaultHandleModalClose;

  console.log('[services]', services);

  const [formData, setFormData] = useState({
    page_heading: "",
    enable_review: false,
    enable_faq: false,
    banner: {
      banner_title_heading: "",
      banner_text: "",
      background_image_url: ""
    },
    stand_out_section: {
      section_span: "",
      section_text: "",
      background_file_url: ""
    },
    our_services_section: {
      section_heading: "",
      services_cards: [
        {
          icon_url: "",
          title: "",
          description: ""
        }
      ]
    },
    our_experties: {
      section_heading: "",
      section_span: "",
      section_text: "",
      sections_cards: {
        sections_heading: "",
        sections_cards: [
          {
            icon_url: "",
            title: "",
            description: ""
          }
        ]
      }
    },
    technologies_we_use: {
      section_heading: "",
      section_text: "",
      section_span: "",
      section_background: "",
      card: {
        section_heading: "",
        cards: [
          {
            title: "",
            description: "",
            image: ""
          }
        ]
      }
    },
    services_cards: {
      section_heading: "",
      section_text: "",
      section_span: "",
      section_cards: [
        {
          title: "",
          heading: "",
          text: "",
          image: ""
        }
      ]
    },
    view_profile: {
      section_heading: "",
      button_span: "",
      button_text: "",
      button_url: "",
      section_background: "",
      image: ""
    },
    cards: [
      {
        heading: "",
        image: "",
        backgroud: ""
      }
    ]
  });

  // Pre-fill form data when in edit mode
  useEffect(() => {
    if (isEdit && initialData) {
      // Ensure all sections have proper structure even if they're null in the API response
      const processedData = {
        ...initialData,
        banner: initialData.banner || {
          banner_title_heading: "",
          banner_text: "",
          background_image_url: ""
        },
        stand_out_section: initialData.stand_out_section || {
          section_span: "",
          section_text: "",
          background_file_url: ""
        },
        our_services_section: initialData.our_services_section || {
          section_heading: "",
          services_cards: [
            {
              icon_url: "",
              title: "",
              description: ""
            }
          ]
        },
        our_experties: initialData.our_experties || {
          section_heading: "",
          section_span: "",
          section_text: "",
          sections_cards: {
            sections_heading: "",
            sections_cards: [
              {
                icon_url: "",
                title: "",
                description: ""
              }
            ]
          }
        },
        technologies_we_use: initialData.technologies_we_use || {
          section_heading: "",
          section_text: "",
          section_span: "",
          section_background: "",
          card: {
            section_heading: "",
            cards: [
              {
                title: "",
                description: "",
                image: ""
              }
            ]
          }
        },
        services_cards: initialData.services_cards || {
          section_heading: "",
          section_text: "",
          section_span: "",
          section_cards: [
            {
              title: "",
              heading: "",
              text: "",
              image: ""
            }
          ]
        },
        view_profile: initialData.view_profile || {
          section_heading: "",
          button_span: "",
          button_text: "",
          button_url: "",
          section_background: "",
          image: ""
        },
        cards: initialData.cards || [
          {
            heading: "",
            image: "",
            backgroud: ""
          }
        ]
      };
      setFormData(processedData);
    }
  }, [isEdit, initialData]);

  const handleSubmit = (event) => {
    event.preventDefault();
    console.log('Form Data:', formData);
    if (formData.page_heading) {
      if (isEdit) {
        console.log('initialData', initialData)
        console.log('formData', formData)

        dispatch(UpdateSeoServiceTemplatePage({ id: initialData.id, data: formData }))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Updated');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      } else {
        dispatch(SeoTemplatePage(formData))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Created');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      }
    } else {
      toast.error('Page Heading is required');
    }
  };

  const handleInputChange = (section, subsection, index, field) => {
    if (section && subsection && index !== null && field) {
      // Handle nested array item field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? { ...item, [field]: event.target.value } : item
          )
        }
      }));
    } else if (section && subsection && index !== null) {
      // Handle array item (for services array)
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? event.target.value : item
          )
        }
      }));
    } else if (section && subsection) {
      // Handle nested object field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: event.target.value
        }
      }));
    } else if (section) {
      // Handle direct field
      setFormData(prev => ({
        ...prev,
        [section]: event.target.value
      }));
    }
  };

  const handleSwitchChange = (field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.checked
    }));
  };

  const handleFileChange = async (section, subsection, index, field) => {
    const file = event.target.files[0];
    if (!file) return;

    // Create a unique key for this file upload
    const uploadKey = `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
    
    // Set uploading state
    setUploadingFiles(prev => ({ ...prev, [uploadKey]: true }));
    setUploadErrors(prev => ({ ...prev, [uploadKey]: null }));

    try {
      const result = await dispatch(UploadServiceMediaAction({ file })).unwrap();
      
      if (result.success && result.data?.file_url) {
        // Update form data with the returned file_url
        if (section && subsection && index !== null && field) {
          // Handle nested array item field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? { ...item, [field]: result.data.file_url } : item
              )
            }
          }));
        } else if (section && subsection) {
          // Handle nested object field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: result.data.file_url
            }
          }));
        } else if (section) {
          // Handle direct field
          setFormData(prev => ({
            ...prev,
            [section]: result.data.file_url
          }));
        }
      }
    } catch (error) {
      console.error('File upload failed:', error);
      setUploadErrors(prev => ({ 
        ...prev, 
        [uploadKey]: error || 'File upload failed' 
      }));
    } finally {
      setUploadingFiles(prev => ({ ...prev, [uploadKey]: false }));
    }
  };

  const getUploadKey = (section, subsection, index, field) => {
    return `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
  };

  const isUploading = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadingFiles[key] || false;
  };

  const getUploadError = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadErrors[key] || null;
  };

  const addServiceCard = () => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services_cards: [...prev.our_services_section.services_cards, { icon_url: "", title: "", description: "" }]
      }
    }));
  };

  const removeServiceCard = (index) => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services_cards: prev.our_services_section.services_cards.filter((_, i) => i !== index)
      }
    }));
  };

  const addExpertiseCard = () => {
    setFormData(prev => ({
      ...prev,
      our_experties: {
        ...prev.our_experties,
        sections_cards: {
          ...prev.our_experties.sections_cards,
          sections_cards: [...prev.our_experties.sections_cards.sections_cards, { icon_url: "", title: "", description: "" }]
        }
      }
    }));
  };

  const removeExpertiseCard = (index) => {
    setFormData(prev => ({
      ...prev,
      our_experties: {
        ...prev.our_experties,
        sections_cards: {
          ...prev.our_experties.sections_cards,
          sections_cards: prev.our_experties.sections_cards.sections_cards.filter((_, i) => i !== index)
        }
      }
    }));
  };

  const addTechnologyCard = () => {
    setFormData(prev => ({
      ...prev,
      technologies_we_use: {
        ...prev.technologies_we_use,
        card: {
          ...prev.technologies_we_use.card,
          cards: [...prev.technologies_we_use.card.cards, { title: "", description: "", image: "" }]
        }
      }
    }));
  };

  const removeTechnologyCard = (index) => {
    setFormData(prev => ({
      ...prev,
      technologies_we_use: {
        ...prev.technologies_we_use,
        card: {
          ...prev.technologies_we_use.card,
          cards: prev.technologies_we_use.card.cards.filter((_, i) => i !== index)
        }
      }
    }));
  };

  const addServiceCardItem = () => {
    setFormData(prev => ({
      ...prev,
      services_cards: {
        ...prev.services_cards,
        section_cards: [...prev.services_cards.section_cards, { title: "", heading: "", text: "", image: "" }]
      }
    }));
  };

  const removeServiceCardItem = (index) => {
    setFormData(prev => ({
      ...prev,
      services_cards: {
        ...prev.services_cards,
        section_cards: prev.services_cards.section_cards.filter((_, i) => i !== index)
      }
    }));
  };

  const addCard = () => {
    setFormData(prev => ({
      ...prev,
      cards: [...prev.cards, { heading: "", image: "", backgroud: "" }]
    }));
  };

  const removeCard = (index) => {
    setFormData(prev => ({
      ...prev,
      cards: prev.cards.filter((_, i) => i !== index)
    }));
  };



  return (
    <Container maxWidth="xl" sx={{ py: 4 }}>
      <Paper elevation={3} sx={{ p: 4 }}>

        <Box component="form" onSubmit={handleSubmit}>
          {/* Page Heading */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Heading
              </Typography>
              <TextField
                fullWidth
                label="Page Heading"
                value={formData.page_heading}
                onChange={(e) => handleInputChange('page_heading', null, null, null)}
                placeholder="Enter page heading"
                variant="outlined"
                margin="normal"
              />
            </CardContent>
          </Card>

          {/* Banner Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Banner Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Title Heading"
                    value={formData.banner.banner_title_heading}
                    onChange={(e) => handleInputChange('banner', 'banner_title_heading', null, null)}
                    placeholder="Enter banner title heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Text"
                    value={formData.banner.banner_text}
                    onChange={(e) => handleInputChange('banner', 'banner_text', null, null)}
                    placeholder="Enter banner text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('banner', 'background_image_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('banner', 'background_image_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('banner', 'background_image_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('banner', 'background_image_url', null, null)}</Alert>
                  )}
                  {formData.banner.background_image_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.banner.background_image_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Stand Out Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Stand Out Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Span"
                    value={formData.stand_out_section.section_span}
                    onChange={(e) => handleInputChange('stand_out_section', 'section_span', null, null)}
                    placeholder="Enter section span"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.stand_out_section.section_text}
                    onChange={(e) => handleInputChange('stand_out_section', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background File
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('stand_out_section', 'background_file_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('stand_out_section', 'background_file_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('stand_out_section', 'background_file_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('stand_out_section', 'background_file_url', null, null)}</Alert>
                  )}
                  {formData.stand_out_section.background_file_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.stand_out_section.background_file_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Our Services Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Our Services Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addServiceCard}
                  
                >
                  Add Service Card
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.our_services_section.section_heading}
                onChange={(e) => handleInputChange('our_services_section', 'section_heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.our_services_section.services_cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Service Card {index + 1}</Typography>
                      {formData.our_services_section.services_cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeServiceCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_services_section', 'services_cards', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_services_section', 'services_cards', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_services_section', 'services_cards', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_services_section', 'services_cards', index, 'icon_url')}</Alert>
                        )}
                        {card.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={card.title}
                          onChange={(e) => handleInputChange('our_services_section', 'services_cards', index, 'title')}
                          placeholder="Enter service title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={card.description}
                          onChange={(e) => handleInputChange('our_services_section', 'services_cards', index, 'description')}
                          placeholder="Enter service description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Our Expertise Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Our Expertise
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addExpertiseCard}
                  
                >
                  Add Expertise Card
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.our_experties.section_heading}
                onChange={(e) => handleInputChange('our_experties', 'section_heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Span"
                value={formData.our_experties.section_span}
                onChange={(e) => handleInputChange('our_experties', 'section_span', null, null)}
                placeholder="Enter section span"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Text"
                value={formData.our_experties.section_text}
                onChange={(e) => handleInputChange('our_experties', 'section_text', null, null)}
                placeholder="Enter section text"
                variant="outlined"
                multiline
                rows={3}
                margin="normal"
                sx={{ mb: 3 }}
              />
              <TextField
                fullWidth
                label="Sections Cards Heading"
                value={formData.our_experties.sections_cards.sections_heading}
                onChange={(e) => handleInputChange('our_experties', 'sections_cards', null, 'sections_heading')}
                placeholder="Enter sections cards heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.our_experties.sections_cards.sections_cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Expertise Card {index + 1}</Typography>
                      {formData.our_experties.sections_cards.sections_cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeExpertiseCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_experties', 'sections_cards', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_experties', 'sections_cards', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_experties', 'sections_cards', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_experties', 'sections_cards', index, 'icon_url')}</Alert>
                        )}
                        {card.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={card.title}
                          onChange={(e) => handleInputChange('our_experties', 'sections_cards', index, 'title')}
                          placeholder="Enter expertise title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={card.description}
                          onChange={(e) => handleInputChange('our_experties', 'sections_cards', index, 'description')}
                          placeholder="Enter expertise description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Technologies We Use Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Technologies We Use
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addTechnologyCard}
                  
                >
                  Add Technology Card
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.technologies_we_use.section_heading}
                onChange={(e) => handleInputChange('technologies_we_use', 'section_heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Text"
                value={formData.technologies_we_use.section_text}
                onChange={(e) => handleInputChange('technologies_we_use', 'section_text', null, null)}
                placeholder="Enter section text"
                variant="outlined"
                multiline
                rows={3}
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Span"
                value={formData.technologies_we_use.section_span}
                onChange={(e) => handleInputChange('technologies_we_use', 'section_span', null, null)}
                placeholder="Enter section span"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <Typography variant="subtitle1" gutterBottom sx={{ mb: 2 }}>
                Section Background
              </Typography>
              <input
                type="file"
                accept="image/*"
                onChange={(e) => handleFileChange('technologies_we_use', 'section_background', null, null)}
                style={{ width: '100%', marginBottom: '16px' }}
              />
              {isUploading('technologies_we_use', 'section_background', null, null) && (
                <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                  <CircularProgress size={20} />
                  <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                </Box>
              )}
              {getUploadError('technologies_we_use', 'section_background', null, null) && (
                <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('technologies_we_use', 'section_background', null, null)}</Alert>
              )}
              {formData.technologies_we_use.section_background && (
                <Alert severity="success" sx={{ mt: 1 }}>
                  <Typography variant="body2">
                    <strong>Uploaded:</strong> {formData.technologies_we_use.section_background}
                  </Typography>
                </Alert>
              )}
              <TextField
                fullWidth
                label="Card Section Heading"
                value={formData.technologies_we_use.card.section_heading}
                onChange={(e) => handleInputChange('technologies_we_use', 'card', null, 'section_heading')}
                placeholder="Enter card section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.technologies_we_use.card.cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Technology Card {index + 1}</Typography>
                      {formData.technologies_we_use.card.cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeTechnologyCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={card.title}
                          onChange={(e) => handleInputChange('technologies_we_use', 'card', index, 'title')}
                          placeholder="Enter technology title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={card.description}
                          onChange={(e) => handleInputChange('technologies_we_use', 'card', index, 'description')}
                          placeholder="Enter technology description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Image
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('technologies_we_use', 'card', index, 'image')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('technologies_we_use', 'card', index, 'image') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('technologies_we_use', 'card', index, 'image') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('technologies_we_use', 'card', index, 'image')}</Alert>
                        )}
                        {card.image && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.image}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Services Cards Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Services Cards
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addServiceCardItem}
                  
                >
                  Add Service Card Item
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.services_cards.section_heading}
                onChange={(e) => handleInputChange('services_cards', 'section_heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Text"
                value={formData.services_cards.section_text}
                onChange={(e) => handleInputChange('services_cards', 'section_text', null, null)}
                placeholder="Enter section text"
                variant="outlined"
                multiline
                rows={3}
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Span"
                value={formData.services_cards.section_span}
                onChange={(e) => handleInputChange('services_cards', 'section_span', null, null)}
                placeholder="Enter section span"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.services_cards.section_cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Service Card Item {index + 1}</Typography>
                      {formData.services_cards.section_cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeServiceCardItem(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={card.title}
                          onChange={(e) => handleInputChange('services_cards', 'section_cards', index, 'title')}
                          placeholder="Enter title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Heading"
                          value={card.heading}
                          onChange={(e) => handleInputChange('services_cards', 'section_cards', index, 'heading')}
                          placeholder="Enter heading"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Text"
                          value={card.text}
                          onChange={(e) => handleInputChange('services_cards', 'section_cards', index, 'text')}
                          placeholder="Enter text"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Image
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('services_cards', 'section_cards', index, 'image')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('services_cards', 'section_cards', index, 'image') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('services_cards', 'section_cards', index, 'image') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('services_cards', 'section_cards', index, 'image')}</Alert>
                        )}
                        {card.image && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.image}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* View Profile Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                View Profile Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.view_profile.section_heading}
                    onChange={(e) => handleInputChange('view_profile', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={4}>
                  <TextField
                    fullWidth
                    label="Button Span"
                    value={formData.view_profile.button_span}
                    onChange={(e) => handleInputChange('view_profile', 'button_span', null, null)}
                    placeholder="Enter button span"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={4}>
                  <TextField
                    fullWidth
                    label="Button Text"
                    value={formData.view_profile.button_text}
                    onChange={(e) => handleInputChange('view_profile', 'button_text', null, null)}
                    placeholder="Enter button text"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={4}>
                  <TextField
                    fullWidth
                    label="Button URL"
                    value={formData.view_profile.button_url}
                    onChange={(e) => handleInputChange('view_profile', 'button_url', null, null)}
                    placeholder="Enter button URL"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={6}>
                  <Typography variant="subtitle1" gutterBottom>
                    Section Background
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('view_profile', 'section_background', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('view_profile', 'section_background', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('view_profile', 'section_background', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('view_profile', 'section_background', null, null)}</Alert>
                  )}
                  {formData.view_profile.section_background && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.view_profile.section_background}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
                <Grid item xs={12} md={6}>
                  <Typography variant="subtitle1" gutterBottom>
                    Profile Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('view_profile', 'image', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('view_profile', 'image', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('view_profile', 'image', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('view_profile', 'image', null, null)}</Alert>
                  )}
                  {formData.view_profile.image && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.view_profile.image}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Cards Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Cards Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addCard}
                  
                >
                  Add Card
                </Button>
              </Box>
              <Stack spacing={3}>
                {formData.cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Card {index + 1}</Typography>
                      {formData.cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Heading"
                          value={card.heading}
                          onChange={(e) => handleInputChange('cards', null, index, 'heading')}
                          placeholder="Enter heading"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Image"
                          value={card.image}
                          onChange={(e) => handleInputChange('cards', null, index, 'image')}
                          placeholder="Enter image"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Background
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('cards', null, index, 'backgroud')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('cards', null, index, 'backgroud') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('cards', null, index, 'backgroud') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('cards', null, index, 'backgroud')}</Alert>
                        )}
                        {card.backgroud && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.backgroud}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Page Settings Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Settings
              </Typography>
              <Box sx={{ display: 'flex', gap: 4, flexWrap: 'wrap' }}>
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_review}
                      onChange={handleSwitchChange('enable_review')}
                      color="primary"
                    />
                  }
                  label="Enable Review on this page"
                />
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_faq}
                      onChange={handleSwitchChange('enable_faq')}
                      color="primary"
                    />
                  }
                  label="Enable FAQ on this page"
                />
              </Box>
            </CardContent>
          </Card>

          {/* Submit Button */}
          <Box display="flex" justifyContent="center" mt={4}>
            <Button
              type="submit"
              variant="contained"
              size="large"
              
              sx={{ py: 1, px: 4, fontSize: '1rem' }}
            >
              {isEdit ? 'Update Page' : 'Submit Form'}
            </Button>
          </Box>
        </Box>
      </Paper>
    </Container>
  );
};

export default SeoServiceForm;