import React, { useState, useEffect } from 'react';
import {
  Box,
  TextField,
  Button,
  Typography,
  Paper,
  Grid,
  Card,
  CardContent,
  IconButton,
  Divider,
  Container,
  Stack,
  CircularProgress,
  Alert,
  Chip,
  Switch,
  FormControlLabel
} from '@mui/material';
import { Add as AddIcon, Delete as DeleteIcon } from '@mui/icons-material';
import { useSelector, useDispatch } from 'react-redux';
import { GetAllServicesPages, SaveSearchEngineMarketingTemplatePage, UploadServiceMediaAction, UpdateSearchEngineMarkettingTemplatePage } from '../../store/slices/servicesSlice/servicesSlice';
import { toast } from 'react-toastify';
import { GetAllServicesTabs } from '../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { useModal } from '../../hooks/useModal';

const SearchEngineMarketingServiceForm = ({ isEdit, initialData, onSuccess, handleModalClose, setIsEdit }) => {
  const { handleModalClose: defaultHandleModalClose } = useModal()
  const { services } = useSelector(state => state.ServicesReducer);
  const dispatch = useDispatch();
  const [uploadingFiles, setUploadingFiles] = useState({});
  const [uploadErrors, setUploadErrors] = useState({});

  // Use passed handleModalClose if available, otherwise use default
  const modalCloseHandler = handleModalClose || defaultHandleModalClose;

  const [formData, setFormData] = useState({
    page_heading: "",
    enable_review: false,
    enable_faq: false,
    banner: {
      banner_title_heading: "",
      banner_text: "",
      background_image_url: ""
    },
    stand_out_section: {
      section_span: "",
      section_text: "",
      background_file_url: "",
      image: [""]
    },
    our_achievements_section: {
      section_headign: "",
      section_text: "",
      features: [
        {
          icon_url: "",
          title: "",
          text: ""
        }
      ]
    },
    our_services_section: {
      section_heading: "",
      section_background: "",
      services_cards: [
        {
          icon_url: "",
          title: "",
          description: ""
        }
      ]
    },
    technnologies_section: {
      heading: "",
      image_url: ""
    },
    our_experties: {
      section_heading: "",
      section_background: "",
      section_span: "",
      sections_cards: [
        {
          icon_url: "",
          title: "",
          description: ""
        }
      ]
    },
    about_us: {
      section_heading: "",
      section_text: "",
      button_text: "",
      button_url: "",
      section_background: ""
    }
  });

  // Pre-fill form data when in edit mode
  useEffect(() => {
    if (isEdit && initialData) {
      // Ensure all sections have proper structure even if they're null in the API response
      const processedData = {
        ...initialData,
        banner: initialData.banner || {
          banner_title_heading: "",
          banner_text: "",
          background_image_url: ""
        },
        stand_out_section: initialData.stand_out_section || {
          section_span: "",
          section_text: "",
          background_file_url: "",
          image: [""]
        },
        our_achievements_section: initialData.our_achievements_section || {
          section_headign: "",
          section_text: "",
          features: [
            {
              icon_url: "",
              title: "",
              text: ""
            }
          ]
        },
        our_services_section: initialData.our_services_section || {
          section_heading: "",
          section_background: "",
          services_cards: [
            {
              icon_url: "",
              title: "",
              description: ""
            }
          ]
        },
        technnologies_section: initialData.technnologies_section || {
          heading: "",
          image_url: ""
        },
        our_experties: initialData.our_experties || {
          section_heading: "",
          section_background: "",
          section_span: "",
          sections_cards: [
            {
              icon_url: "",
              title: "",
              description: ""
            }
          ]
        },
        about_us: initialData.about_us || {
          section_heading: "",
          section_text: "",
          button_text: "",
          button_url: "",
          section_background: ""
        }
      };
      setFormData(processedData);
    }
  }, [isEdit, initialData]);

  const handleSubmit = (event) => {
    event.preventDefault();
    console.log('Form Data:', formData);
    if (formData.page_heading) {
      if (isEdit) {
        console.log('initialData', initialData)
        console.log('formData', formData)

        dispatch(UpdateSearchEngineMarkettingTemplatePage({ id: initialData.id, data: formData }))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Updated');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      } else {
        dispatch(SaveSearchEngineMarketingTemplatePage(formData))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Created');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      }
    } else {
      toast.error('Page Heading is required');
    }
  };

  const handleInputChange = (section, subsection, index, field) => {
    if (section && subsection && index !== null && field) {
      // Handle nested array item field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? { ...item, [field]: event.target.value } : item
          )
        }
      }));
    } else if (section && subsection && index !== null) {
      // Handle array item (for simple arrays)
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? event.target.value : item
          )
        }
      }));
    } else if (section && subsection) {
      // Handle nested object field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: event.target.value
        }
      }));
    } else if (section) {
      // Handle direct field
      setFormData(prev => ({
        ...prev,
        [section]: event.target.value
      }));
    }
  };

  const handleFileChange = async (section, subsection, index, field) => {
    const file = event.target.files[0];
    if (!file) return;

    const uploadKey = `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
    
    setUploadingFiles(prev => ({ ...prev, [uploadKey]: true }));
    setUploadErrors(prev => ({ ...prev, [uploadKey]: null }));

    try {
      const result = await dispatch(UploadServiceMediaAction({ file })).unwrap();
      
      if (result.success && result.data?.file_url) {
        if (section && subsection && index !== null && field) {
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? { ...item, [field]: result.data.file_url } : item
              )
            }
          }));
        } else if (section && subsection) {
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: result.data.file_url
            }
          }));
        } else if (section) {
          setFormData(prev => ({
            ...prev,
            [section]: result.data.file_url
          }));
        }
      }
    } catch (error) {
      console.error('File upload failed:', error);
      setUploadErrors(prev => ({ 
        ...prev, 
        [uploadKey]: error || 'File upload failed' 
      }));
    } finally {
      setUploadingFiles(prev => ({ ...prev, [uploadKey]: false }));
    }
  };

  const handleSwitchChange = (field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.checked
    }));
  };

  const getUploadKey = (section, subsection, index, field) => {
    return `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
  };

  const isUploading = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadingFiles[key] || false;
  };

  const getUploadError = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadErrors[key] || null;
  };

  // Add/Remove functions for different sections
  const addStandOutImage = () => {
    setFormData(prev => ({
      ...prev,
      stand_out_section: {
        ...prev.stand_out_section,
        image: [...prev.stand_out_section.image, ""]
      }
    }));
  };

  const removeStandOutImage = (index) => {
    setFormData(prev => ({
      ...prev,
      stand_out_section: {
        ...prev.stand_out_section,
        image: prev.stand_out_section.image.filter((_, i) => i !== index)
      }
    }));
  };

  const addAchievement = () => {
    setFormData(prev => ({
      ...prev,
      our_achievements_section: {
        ...prev.our_achievements_section,
        features: [...prev.our_achievements_section.features, { icon_url: "", title: "", text: "" }]
      }
    }));
  };

  const removeAchievement = (index) => {
    setFormData(prev => ({
      ...prev,
      our_achievements_section: {
        ...prev.our_achievements_section,
        features: prev.our_achievements_section.features.filter((_, i) => i !== index)
      }
    }));
  };

  const addServiceCard = () => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services_cards: [...prev.our_services_section.services_cards, { icon_url: "", title: "", description: "" }]
      }
    }));
  };

  const removeServiceCard = (index) => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services_cards: prev.our_services_section.services_cards.filter((_, i) => i !== index)
      }
    }));
  };

  const addExpertiseCard = () => {
    setFormData(prev => ({
      ...prev,
      our_experties: {
        ...prev.our_experties,
        sections_cards: [...prev.our_experties.sections_cards, { icon_url: "", title: "", description: "" }]
      }
    }));
  };

  const removeExpertiseCard = (index) => {
    setFormData(prev => ({
      ...prev,
      our_experties: {
        ...prev.our_experties,
        sections_cards: prev.our_experties.sections_cards.filter((_, i) => i !== index)
      }
    }));
  };



  return (
    <Container maxWidth="xl" sx={{ py: 4 }}>
      <Paper elevation={3} sx={{ p: 4 }}>
        <Box component="form" onSubmit={handleSubmit}>
          {/* Page Heading */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Page Heading
              </Typography>
              <TextField
                fullWidth
                label="Page Heading"
                value={formData.page_heading}
                onChange={(e) => handleInputChange('page_heading', null, null, null)}
                placeholder="Enter page heading"
                variant="outlined"
                margin="normal"
              />
            </CardContent>
          </Card>

          {/* Banner Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Banner Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Title Heading"
                    value={formData.banner.banner_title_heading}
                    onChange={(e) => handleInputChange('banner', 'banner_title_heading', null, null)}
                    placeholder="Enter banner title heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Text"
                    value={formData.banner.banner_text}
                    onChange={(e) => handleInputChange('banner', 'banner_text', null, null)}
                    placeholder="Enter banner text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('banner', 'background_image_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('banner', 'background_image_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('banner', 'background_image_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('banner', 'background_image_url', null, null)}</Alert>
                  )}
                  {formData.banner.background_image_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.banner.background_image_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Stand Out Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Stand Out Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Span"
                    value={formData.stand_out_section.section_span}
                    onChange={(e) => handleInputChange('stand_out_section', 'section_span', null, null)}
                    placeholder="Enter section span"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.stand_out_section.section_text}
                    onChange={(e) => handleInputChange('stand_out_section', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background File
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('stand_out_section', 'background_file_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('stand_out_section', 'background_file_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('stand_out_section', 'background_file_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('stand_out_section', 'background_file_url', null, null)}</Alert>
                  )}
                  {formData.stand_out_section.background_file_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.stand_out_section.background_file_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
                <Grid item xs={12} md={12}>
                  <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                    <Typography variant="subtitle1">Section Images</Typography>
                    <Button
                      variant="outlined"
                      startIcon={<AddIcon />}
                      onClick={addStandOutImage}
                      size="small"
                    >
                      Add Image
                    </Button>
                  </Box>
                  <Stack spacing={2}>
                    {formData.stand_out_section.image.map((image, index) => (
                      <Box key={index} display="flex" gap={2} alignItems="center">
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('stand_out_section', 'image', index, null)}
                          style={{ flex: 1 }}
                        />
                        {formData.stand_out_section.image.length > 1 && (
                          <IconButton
                            color="error"
                            onClick={() => removeStandOutImage(index)}
                            size="small"
                          >
                            <DeleteIcon />
                          </IconButton>
                        )}
                      </Box>
                    ))}
                  </Stack>
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Our Achievements Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Our Achievements Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addAchievement}
                >
                  Add Achievement
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.our_achievements_section.section_headign}
                    onChange={(e) => handleInputChange('our_achievements_section', 'section_headign', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.our_achievements_section.section_text}
                    onChange={(e) => handleInputChange('our_achievements_section', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.our_achievements_section.features.map((achievement, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Achievement {index + 1}</Typography>
                      {formData.our_achievements_section.features.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeAchievement(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_achievements_section', 'features', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_achievements_section', 'features', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_achievements_section', 'features', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_achievements_section', 'features', index, 'icon_url')}</Alert>
                        )}
                        {achievement.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {achievement.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={achievement.title}
                          onChange={(e) => handleInputChange('our_achievements_section', 'features', index, 'title')}
                          placeholder="Enter achievement title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Text"
                          value={achievement.text}
                          onChange={(e) => handleInputChange('our_achievements_section', 'features', index, 'text')}
                          placeholder="Enter achievement text"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Our Services Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Our Services Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addServiceCard}
                >
                  Add Service Card
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.our_services_section.section_heading}
                    onChange={(e) => handleInputChange('our_services_section', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Section Background
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('our_services_section', 'section_background', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('our_services_section', 'section_background', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('our_services_section', 'section_background', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_services_section', 'section_background', null, null)}</Alert>
                  )}
                  {formData.our_services_section.section_background && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.our_services_section.section_background}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.our_services_section.services_cards.map((service, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Service Card {index + 1}</Typography>
                      {formData.our_services_section.services_cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeServiceCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_services_section', 'services_cards', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_services_section', 'services_cards', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_services_section', 'services_cards', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_services_section', 'services_cards', index, 'icon_url')}</Alert>
                        )}
                        {service.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {service.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={service.title}
                          onChange={(e) => handleInputChange('our_services_section', 'services_cards', index, 'title')}
                          placeholder="Enter service title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={service.description}
                          onChange={(e) => handleInputChange('our_services_section', 'services_cards', index, 'description')}
                          placeholder="Enter service description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Technologies Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Technologies Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Heading"
                    value={formData.technnologies_section.heading}
                    onChange={(e) => handleInputChange('technnologies_section', 'heading', null, null)}
                    placeholder="Enter technologies heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('technnologies_section', 'image_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('technnologies_section', 'image_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('technnologies_section', 'image_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('technnologies_section', 'image_url', null, null)}</Alert>
                  )}
                  {formData.technnologies_section.image_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.technnologies_section.image_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Our Expertise Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Our Expertise Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addExpertiseCard}
                >
                  Add Expertise Card
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.our_experties.section_heading}
                    onChange={(e) => handleInputChange('our_experties', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Background"
                    value={formData.our_experties.section_background}
                    onChange={(e) => handleInputChange('our_experties', 'section_background', null, null)}
                    placeholder="Enter section background"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Span"
                    value={formData.our_experties.section_span}
                    onChange={(e) => handleInputChange('our_experties', 'section_span', null, null)}
                    placeholder="Enter section span"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.our_experties.sections_cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Expertise Card {index + 1}</Typography>
                      {formData.our_experties.sections_cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeExpertiseCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_experties', 'sections_cards', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_experties', 'sections_cards', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_experties', 'sections_cards', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_experties', 'sections_cards', index, 'icon_url')}</Alert>
                        )}
                        {card.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={card.title}
                          onChange={(e) => handleInputChange('our_experties', 'sections_cards', index, 'title')}
                          placeholder="Enter card title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={card.description}
                          onChange={(e) => handleInputChange('our_experties', 'sections_cards', index, 'description')}
                          placeholder="Enter card description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* About Us Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                About Us Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.about_us.section_heading}
                    onChange={(e) => handleInputChange('about_us', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.about_us.section_text}
                    onChange={(e) => handleInputChange('about_us', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={6}>
                  <TextField
                    fullWidth
                    label="Button Text"
                    value={formData.about_us.button_text}
                    onChange={(e) => handleInputChange('about_us', 'button_text', null, null)}
                    placeholder="Enter button text"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={6}>
                  <TextField
                    fullWidth
                    label="Button URL"
                    value={formData.about_us.button_url}
                    onChange={(e) => handleInputChange('about_us', 'button_url', null, null)}
                    placeholder="Enter button URL"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Section Background
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('about_us', 'section_background', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('about_us', 'section_background', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('about_us', 'section_background', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('about_us', 'section_background', null, null)}</Alert>
                  )}
                  {formData.about_us.section_background && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.about_us.section_background}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Page Settings Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Settings
              </Typography>
              <Box sx={{ display: 'flex', gap: 4, flexWrap: 'wrap' }}>
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_review}
                      onChange={handleSwitchChange('enable_review')}
                      color="primary"
                    />
                  }
                  label="Enable Review on this page"
                />
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_faq}
                      onChange={handleSwitchChange('enable_faq')}
                      color="primary"
                    />
                  }
                  label="Enable FAQ on this page"
                />
              </Box>
            </CardContent>
          </Card>

          {/* Submit Button */}
          <Box display="flex" justifyContent="center" mt={4}>
            <Button
              type="submit"
              variant="contained"
              size="large"
              sx={{ py: 1, px: 4, fontSize: '1rem' }}
            >
              {isEdit ? 'Update Page' : 'Submit Form'}
            </Button>
          </Box>
        </Box>
      </Paper>
    </Container>
  );
};

export default SearchEngineMarketingServiceForm;