import React, { useState, useEffect } from 'react';
import {
  Box,
  TextField,
  Button,
  Typography,
  Paper,
  Grid,
  Card,
  CardContent,
  IconButton,
  Divider,
  Container,
  Stack,
  CircularProgress,
  Alert,
  Switch,
  FormControlLabel
} from '@mui/material';
import { Add as AddIcon, Delete as DeleteIcon } from '@mui/icons-material';
import { useSelector, useDispatch } from 'react-redux';
import { GetAllServicesPages, SaveListingTemplatePage, UploadServiceMediaAction, UpdateListingServiceTemplatePage } from '../../store/slices/servicesSlice/servicesSlice';
import { toast } from 'react-toastify';
import { GetAllServicesTabs } from '../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { useModal } from '../../hooks/useModal';

const ListingServiceForm = ({ isEdit, initialData, onSuccess, setIsEdit }) => {
  const { handleModalClose: defaultHandleModalClose } = useModal()
  const { services } = useSelector(state => state.ServicesReducer);
  const dispatch = useDispatch();
  const [uploadingFiles, setUploadingFiles] = useState({});
  const [uploadErrors, setUploadErrors] = useState({});

  // Use passed handleModalClose if available, otherwise use default
  const modalCloseHandler =  defaultHandleModalClose;

  const [formData, setFormData] = useState({
    page_heading: "",
    enable_review: false,
    enable_faq: false,
    banner: {
      banner_title_heading: "",
      banner_text: "",
      background_image_url: ""
    },
    stand_out_section: {
      section_span: "",
      section_text: "",
      background_file_url: "",
      image: ""
    },
    our_achievements_section: {
      section_heading: "",
      section_text: "",
      features: [
        {
          icon_url: "",
          title: "",
          text: ""
        }
      ]
    },
    our_services_section: {
      section_heading: "",
      section_text: "",
      section_background: "",
      services: [
        {
          title: "",
          icon_url: "",
          description: ""
        }
      ]
    },
    technologies_we_work_section: {
      section_heading: "",
      section_text: "",
      technologies: [
        {
          technology_heading: "",
          technologies: [
            {
              image_url: "",
              title: "",
              description: ""
            }
          ]
        }
      ]
    },
    certification_sections: {
      section_heading: "",
      section_span: "",
      certifications: [
        {
          icon_url: "",
          title: ""
        }
      ]
    },
    our_top_clients_section: {
      section_heading: "",
      section_text: "",
      clients: [
        {
          image_url: "",
          name: "",
          description: "",
          position: ""
        }
      ]
    },
    portfolio_section: {
      section_span: "",
      section_heading: "",
      projects: [
        {
          image_url: "",
          title: "",
          description: ""
        }
      ]
    }
  });

  // Pre-fill form data when in edit mode
  useEffect(() => {
    if (isEdit && initialData) {
      // Ensure all sections have proper structure even if they're null in the API response
      const processedData = {
        ...initialData,
        banner: initialData.banner || {
          banner_title_heading: "",
          banner_text: "",
          background_image_url: ""
        },
        stand_out_section: initialData.stand_out_section || {
          section_span: "",
          section_text: "",
          background_file_url: "",
          image: ""
        },
        our_achievements_section: initialData.our_achievements_section || {
          section_heading: "",
          section_text: "",
          features: [
            {
              icon_url: "",
              title: "",
              text: ""
            }
          ]
        },
        our_services_section: initialData.our_services_section || {
          section_heading: "",
          section_text: "",
          section_background: "",
          services: [
            {
              title: "",
              icon_url: "",
              description: ""
            }
          ]
        },
        technologies_we_work_section: JSON.parse(initialData.technologies_we_work_section) || {
          section_heading: "",
          section_text: "",
          technologies: [
            {
              technology_heading: "",
              technologies: [
                {
                  image_url: "",
                  title: "",
                  description: ""
                }
              ]
            }
          ]
        },
        certification_sections: JSON.parse(initialData.certification_sections) || {
          section_heading: "",
          section_span: "",
          certifications: [
            {
              icon_url: "",
              title: ""
            }
          ]
        },
        our_top_clients_section: initialData.our_top_clients_section || {
          section_heading: "",
          section_text: "",
          clients: [
            {
              image_url: "",
              name: "",
              description: "",
              position: ""
            }
          ]
        },
        portfolio_section: initialData.portfolio_section || {
          section_span: "",
          section_heading: "",
          projects: [
            {
              image_url: "",
              title: "",
              description: ""
            }
          ]
        }
      };
      setFormData(processedData);
    }
  }, [isEdit, initialData]);

  const handleSubmit = (event) => {
    event.preventDefault();
    console.log('Form Data:', formData);
    if (formData.page_heading) {
      if (isEdit) {
        console.log('initialData', initialData)
        console.log('formData', formData)

        dispatch(UpdateListingServiceTemplatePage({ id: initialData.id, data: formData }))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Updated');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      } else {
        dispatch(SaveListingTemplatePage(formData))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Created');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      }
    } else {
      toast.error('Page Heading is required');
    }
  };

  const handleInputChange = (section, subsection, index, field) => {
    if (section && subsection && index !== null && field) {
      // Handle nested array item field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? { ...item, [field]: event.target.value } : item
          )
        }
      }));
    } else if (section && subsection && index !== null) {
      // Handle array item (for simple arrays)
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? event.target.value : item
          )
        }
      }));
    } else if (section && subsection) {
      // Handle nested object field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: event.target.value
        }
      }));
    } else if (section) {
      // Handle direct field
      setFormData(prev => ({
        ...prev,
        [section]: event.target.value
      }));
    }
  };

  const handleSwitchChange = (field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.checked
    }));
  };

  const handleFileChange = async (section, subsection, index, field) => {
    const file = event.target.files[0];
    if (!file) return;

    const uploadKey = `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
    
    setUploadingFiles(prev => ({ ...prev, [uploadKey]: true }));
    setUploadErrors(prev => ({ ...prev, [uploadKey]: null }));

    try {
      const result = await dispatch(UploadServiceMediaAction({ file })).unwrap();
      
      if (result.success && result.data?.file_url) {
        if (section && subsection && index !== null && field) {
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? { ...item, [field]: result.data.file_url } : item
              )
            }
          }));
        } else if (section && subsection) {
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: result.data.file_url
            }
          }));
        } else if (section) {
          setFormData(prev => ({
            ...prev,
            [section]: result.data.file_url
          }));
        }
      }
    } catch (error) {
      console.error('File upload failed:', error);
      setUploadErrors(prev => ({ 
        ...prev, 
        [uploadKey]: error || 'File upload failed' 
      }));
    } finally {
      setUploadingFiles(prev => ({ ...prev, [uploadKey]: false }));
    }
  };

  const getUploadKey = (section, subsection, index, field) => {
    return `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
  };

  const isUploading = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadingFiles[key] || false;
  };

  const getUploadError = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadErrors[key] || null;
  };

  // Add/Remove functions for different sections
  const addAchievement = () => {
    setFormData(prev => ({
      ...prev,
      our_achievements_section: {
        ...prev.our_achievements_section,
        features: [...prev.our_achievements_section.features, { icon_url: "", title: "", text: "" }]
      }
    }));
  };

  const removeAchievement = (index) => {
    setFormData(prev => ({
      ...prev,
      our_achievements_section: {
        ...prev.our_achievements_section,
        features: prev.our_achievements_section.features.filter((_, i) => i !== index)
      }
    }));
  };

  const addService = () => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services: [...prev.our_services_section.services, { title: "", icon_url: "", description: "" }]
      }
    }));
  };

  const removeService = (index) => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services: prev.our_services_section.services.filter((_, i) => i !== index)
      }
    }));
  };

  const addTechnologySection = () => {
    setFormData(prev => ({
      ...prev,
      technologies_we_work_section: {
        ...prev.technologies_we_work_section,
        technologies: [...prev.technologies_we_work_section.technologies, { 
          technology_heading: "", 
          technologies: [{ image_url: "", title: "", description: "" }] 
        }]
      }
    }));
  };

  const removeTechnologySection = (index) => {
    setFormData(prev => ({
      ...prev,
      technologies_we_work_section: {
        ...prev.technologies_we_work_section,
        technologies: prev.technologies_we_work_section.technologies.filter((_, i) => i !== index)
      }
    }));
  };

  const addTechnology = (sectionIndex) => {
    setFormData(prev => ({
      ...prev,
      technologies_we_work_section: {
        ...prev.technologies_we_work_section,
        technologies: prev.technologies_we_work_section.technologies.map((section, i) =>
          i === sectionIndex 
            ? { ...section, technologies: [...section.technologies, { image_url: "", title: "", description: "" }] }
            : section
        )
      }
    }));
  };

  const removeTechnology = (sectionIndex, techIndex) => {
    setFormData(prev => ({
      ...prev,
      technologies_we_work_section: {
        ...prev.technologies_we_work_section,
        technologies: prev.technologies_we_work_section.technologies.map((section, i) =>
          i === sectionIndex 
            ? { ...section, technologies: section.technologies.filter((_, j) => j !== techIndex) }
            : section
        )
      }
    }));
  };

  const addCertification = () => {
    setFormData(prev => ({
      ...prev,
      certification_sections: {
        ...prev.certification_sections,
        certifications: [...prev.certification_sections.certifications, { icon_url: "", title: "" }]
      }
    }));
  };

  const removeCertification = (index) => {
    setFormData(prev => ({
      ...prev,
      certification_sections: {
        ...prev.certification_sections,
        certifications: prev.certification_sections.certifications.filter((_, i) => i !== index)
      }
    }));
  };

  const addClient = () => {
    setFormData(prev => ({
      ...prev,
      our_top_clients_section: {
        ...prev.our_top_clients_section,
        clients: [...prev.our_top_clients_section.clients, { image_url: "", name: "", description: "", position: "" }]
      }
    }));
  };

  const removeClient = (index) => {
    setFormData(prev => ({
      ...prev,
      our_top_clients_section: {
        ...prev.our_top_clients_section,
        clients: prev.our_top_clients_section.clients.filter((_, i) => i !== index)
      }
    }));
  };

  const addPortfolioProject = () => {
    setFormData(prev => ({
      ...prev,
      portfolio_section: {
        ...prev.portfolio_section,
        projects: [...prev.portfolio_section.projects, { image_url: "", title: "", description: "" }]
      }
    }));
  };

  const removePortfolioProject = (index) => {
    setFormData(prev => ({
      ...prev,
      portfolio_section: {
        ...prev.portfolio_section,
        projects: prev.portfolio_section.projects.filter((_, i) => i !== index)
      }
    }));
  };



  return (
    <Container maxWidth="xl" sx={{ py: 4 }}>
      <Paper elevation={3} sx={{ p: 4 }}>
        <Box component="form" onSubmit={handleSubmit}>
          {/* Page Heading */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Page Heading
              </Typography>
              <TextField
                fullWidth
                label="Page Heading"
                value={formData.page_heading}
                onChange={(e) => handleInputChange('page_heading', null, null, null)}
                placeholder="Enter page heading"
                variant="outlined"
                margin="normal"
              />
            </CardContent>
          </Card>

          {/* Banner Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Banner Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Title Heading"
                    value={formData.banner.banner_title_heading}
                    onChange={(e) => handleInputChange('banner', 'banner_title_heading', null, null)}
                    placeholder="Enter banner title heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Text"
                    value={formData.banner.banner_text}
                    onChange={(e) => handleInputChange('banner', 'banner_text', null, null)}
                    placeholder="Enter banner text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('banner', 'background_image_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('banner', 'background_image_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('banner', 'background_image_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('banner', 'background_image_url', null, null)}</Alert>
                  )}
                  {formData.banner.background_image_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.banner.background_image_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Stand Out Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Stand Out Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Span"
                    value={formData.stand_out_section.section_span}
                    onChange={(e) => handleInputChange('stand_out_section', 'section_span', null, null)}
                    placeholder="Enter section span"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.stand_out_section.section_text}
                    onChange={(e) => handleInputChange('stand_out_section', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={6}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background File
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('stand_out_section', 'background_file_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('stand_out_section', 'background_file_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('stand_out_section', 'background_file_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('stand_out_section', 'background_file_url', null, null)}</Alert>
                  )}
                  {formData.stand_out_section.background_file_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.stand_out_section.background_file_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
                <Grid item xs={12} md={6}>
                  <Typography variant="subtitle1" gutterBottom>
                    Section Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('stand_out_section', 'image', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('stand_out_section', 'image', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('stand_out_section', 'image', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('stand_out_section', 'image', null, null)}</Alert>
                  )}
                  {formData.stand_out_section.image && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.stand_out_section.image}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Our Achievements Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Our Achievements Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addAchievement}
                >
                  Add Achievement
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.our_achievements_section.section_heading}
                    onChange={(e) => handleInputChange('our_achievements_section', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.our_achievements_section.section_text}
                    onChange={(e) => handleInputChange('our_achievements_section', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.our_achievements_section.features && formData.our_achievements_section.features.map((achievement, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Achievement {index + 1}</Typography>
                      {formData.our_achievements_section.features.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeAchievement(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_achievements_section', 'features', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_achievements_section', 'features', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_achievements_section', 'features', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_achievements_section', 'features', index, 'icon_url')}</Alert>
                        )}
                        {achievement.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {achievement.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={achievement.title}
                          onChange={(e) => handleInputChange('our_achievements_section', 'features', index, 'title')}
                          placeholder="Enter achievement title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Text"
                          value={achievement.text}
                          onChange={(e) => handleInputChange('our_achievements_section', 'features', index, 'text')}
                          placeholder="Enter achievement text"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Our Services Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Our Services Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addService}
                >
                  Add Service
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.our_services_section.section_heading}
                    onChange={(e) => handleInputChange('our_services_section', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.our_services_section.section_text}
                    onChange={(e) => handleInputChange('our_services_section', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Section Background
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('our_services_section', 'section_background', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('our_services_section', 'section_background', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('our_services_section', 'section_background', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_services_section', 'section_background', null, null)}</Alert>
                  )}
                  {formData.our_services_section.section_background && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.our_services_section.section_background}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.our_services_section.services && formData.our_services_section.services.map((service, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Service {index + 1}</Typography>
                      {formData.our_services_section.services.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeService(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_services_section', 'services', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_services_section', 'services', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_services_section', 'services', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_services_section', 'services', index, 'icon_url')}</Alert>
                        )}
                        {service.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {service.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={service.title}
                          onChange={(e) => handleInputChange('our_services_section', 'services', index, 'title')}
                          placeholder="Enter service title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={service.description}
                          onChange={(e) => handleInputChange('our_services_section', 'services', index, 'description')}
                          placeholder="Enter service description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Technologies We Work Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Technologies We Work Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addTechnologySection}
                >
                  Add Technology Section
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.technologies_we_work_section.section_heading}
                    onChange={(e) => handleInputChange('technologies_we_work_section', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.technologies_we_work_section.section_text}
                    onChange={(e) => handleInputChange('technologies_we_work_section', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.technologies_we_work_section.technologies && formData.technologies_we_work_section.technologies.map((section, sectionIndex) => (
                  <Card key={sectionIndex} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Technology Section {sectionIndex + 1}</Typography>
                      {formData.technologies_we_work_section.technologies.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeTechnologySection(sectionIndex)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <TextField
                      fullWidth
                      label="Technology Heading"
                      value={section.technology_heading}
                      onChange={(e) => handleInputChange('technologies_we_work_section', 'technologies', sectionIndex, 'technology_heading')}
                      placeholder="Enter technology heading"
                      variant="outlined"
                      margin="normal"
                      sx={{ mb: 3 }}
                    />
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="subtitle1">Technologies</Typography>
                      <Button
                        variant="outlined"
                        startIcon={<AddIcon />}
                        onClick={() => addTechnology(sectionIndex)}
                        size="small"
                      >
                        Add Technology
                      </Button>
                    </Box>
                    <Stack spacing={2}>
                      {section.technologies && section.technologies.map((tech, techIndex) => (
                        <Card key={techIndex} variant="outlined" sx={{ p: 2, bgcolor: 'white' }}>
                          <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                            <Typography variant="subtitle2">Technology {techIndex + 1}</Typography>
                            {section.technologies.length > 1 && (
                              <IconButton
                                color="error"
                                onClick={() => removeTechnology(sectionIndex, techIndex)}
                                size="small"
                              >
                                <DeleteIcon />
                              </IconButton>
                            )}
                          </Box>
                          <Grid container spacing={2}>
                            <Grid item xs={12}>
                              <Typography variant="subtitle2" gutterBottom>
                                Image
                              </Typography>
                              <input
                                type="file"
                                accept="image/*"
                                onChange={(e) => handleFileChange('technologies_we_work_section', 'technologies', sectionIndex, 'technologies')}
                                style={{ width: '100%' }}
                              />
                              {isUploading('technologies_we_work_section', 'technologies', sectionIndex, 'technologies') && (
                                <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                                  <CircularProgress size={20} />
                                  <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                                </Box>
                              )}
                              {getUploadError('technologies_we_work_section', 'technologies', sectionIndex, 'technologies') && (
                                <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('technologies_we_work_section', 'technologies', sectionIndex, 'technologies')}</Alert>
                              )}
                              {tech.image_url && (
                                <Alert severity="success" sx={{ mt: 1 }}>
                                  <Typography variant="body2">
                                    <strong>Uploaded:</strong> {tech.image_url}
                                  </Typography>
                                </Alert>
                              )}
                            </Grid>
                            <Grid item xs={12} md={12}>
                              <TextField
                                fullWidth
                                label="Title"
                                value={tech.title}
                                onChange={(e) => handleInputChange('technologies_we_work_section', 'technologies', sectionIndex, 'technologies')}
                                placeholder="Enter technology title"
                                variant="outlined"
                                size="small"
                              />
                            </Grid>
                            <Grid item xs={12} md={12}>
                              <TextField
                                fullWidth
                                label="Description"
                                value={tech.description}
                                onChange={(e) => handleInputChange('technologies_we_work_section', 'technologies', sectionIndex, 'technologies')}
                                placeholder="Enter technology description"
                                variant="outlined"
                                size="small"
                              />
                            </Grid>
                          </Grid>
                        </Card>
                      ))}
                    </Stack>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Certification Sections */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Certification Sections
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addCertification}
                >
                  Add Certification
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.certification_sections.section_heading}
                    onChange={(e) => handleInputChange('certification_sections', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Span"
                    value={formData.certification_sections.section_span}
                    onChange={(e) => handleInputChange('certification_sections', 'section_span', null, null)}
                    placeholder="Enter section span"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.certification_sections.certifications && formData.certification_sections.certifications.map((certification, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Certification {index + 1}</Typography>
                      {formData.certification_sections.certifications.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeCertification(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('certification_sections', 'certifications', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('certification_sections', 'certifications', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('certification_sections', 'certifications', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('certification_sections', 'certifications', index, 'icon_url')}</Alert>
                        )}
                        {certification.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {certification.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={certification.title}
                          onChange={(e) => handleInputChange('certification_sections', 'certifications', index, 'title')}
                          placeholder="Enter certification title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Our Top Clients Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Our Top Clients Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addClient}
                >
                  Add Client
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.our_top_clients_section.section_heading}
                    onChange={(e) => handleInputChange('our_top_clients_section', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.our_top_clients_section.section_text}
                    onChange={(e) => handleInputChange('our_top_clients_section', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.our_top_clients_section.clients && formData.our_top_clients_section.clients.map((client, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Client {index + 1}</Typography>
                      {formData.our_top_clients_section.clients.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeClient(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Image
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('our_top_clients_section', 'clients', index, 'image_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('our_top_clients_section', 'clients', index, 'image_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('our_top_clients_section', 'clients', index, 'image_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_top_clients_section', 'clients', index, 'image_url')}</Alert>
                        )}
                        {client.image_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {client.image_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Name"
                          value={client.name}
                          onChange={(e) => handleInputChange('our_top_clients_section', 'clients', index, 'name')}
                          placeholder="Enter client name"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Position"
                          value={client.position}
                          onChange={(e) => handleInputChange('our_top_clients_section', 'clients', index, 'position')}
                          placeholder="Enter client position"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={client.description}
                          onChange={(e) => handleInputChange('our_top_clients_section', 'clients', index, 'description')}
                          placeholder="Enter client description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Portfolio Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Portfolio Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addPortfolioProject}
                >
                  Add Project
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Span"
                    value={formData.portfolio_section.section_span}
                    onChange={(e) => handleInputChange('portfolio_section', 'section_span', null, null)}
                    placeholder="Enter section span"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.portfolio_section.section_heading}
                    onChange={(e) => handleInputChange('portfolio_section', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.portfolio_section.projects && formData.portfolio_section.projects.map((project, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Project {index + 1}</Typography>
                      {formData.portfolio_section.projects.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removePortfolioProject(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Project Image
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('portfolio_section', 'projects', index, 'image_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('portfolio_section', 'projects', index, 'image_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('portfolio_section', 'projects', index, 'image_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('portfolio_section', 'projects', index, 'image_url')}</Alert>
                        )}
                        {project.image_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {project.image_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={project.title}
                          onChange={(e) => handleInputChange('portfolio_section', 'projects', index, 'title')}
                          placeholder="Enter project title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={project.description}
                          onChange={(e) => handleInputChange('portfolio_section', 'projects', index, 'description')}
                          placeholder="Enter project description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Page Settings Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Settings
              </Typography>
              <Box sx={{ display: 'flex', gap: 4, flexWrap: 'wrap' }}>
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_review}
                      onChange={handleSwitchChange('enable_review')}
                      color="primary"
                    />
                  }
                  label="Enable Review on this page"
                />
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_faq}
                      onChange={handleSwitchChange('enable_faq')}
                      color="primary"
                    />
                  }
                  label="Enable FAQ on this page"
                />
              </Box>
            </CardContent>
          </Card>

          {/* Submit Button */}
          <Box display="flex" justifyContent="center" mt={4}>
            <Button
              type="submit"
              variant="contained"
              size="large"
              sx={{ py: 1, px: 4, fontSize: '1rem' }}
            >
              {isEdit ? 'Update Page' : 'Submit Form'}
            </Button>
          </Box>
        </Box>
      </Paper>
    </Container>
  );
};

export default ListingServiceForm;