import React, { useState, useEffect } from 'react';
import {
  Box,
  TextField,
  Button,
  Typography,
  Paper,
  Grid,
  Card,
  CardContent,
  IconButton,
  Divider,
  Container,
  Stack,
  CircularProgress,
  Alert,
  Switch,
  FormControlLabel
} from '@mui/material';
import { Add as AddIcon, Delete as DeleteIcon } from '@mui/icons-material';
import { useSelector, useDispatch } from 'react-redux';
import { GetAllServicesPages, SaveInternetMarketingTemplatePage, UploadServiceMediaAction, UpdateInternetMarkettingTemplatePage } from '../../store/slices/servicesSlice/servicesSlice';
import { toast } from 'react-toastify';
import { GetAllServicesTabs } from '../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { useModal } from '../../hooks/useModal';

const InternetMarketingServiceForm = ({ isEdit, initialData, onSuccess, setIsEdit }) => {
  const { handleModalClose: defaultHandleModalClose } = useModal()
  const { services } = useSelector(state => state.ServicesReducer);
  const dispatch = useDispatch();
  const [uploadingFiles, setUploadingFiles] = useState({});
  const [uploadErrors, setUploadErrors] = useState({});

  // Use passed handleModalClose if available, otherwise use default
  const modalCloseHandler =  defaultHandleModalClose;

  const [formData, setFormData] = useState({
    page_heading: "",
    enable_review: false,
    enable_faq: false,
    banner: {
      banner_title_heading: "",
      banner_text: "",
      background_image_url: ""
    },
    services_tabs_section_image: {
      text: "",
      services_tabs_section_image: ""
    },
    stand_out_section: {
      section_span: "",
      features: [
        {
          title: "",
          icon_url: ""
        }
      ]
    },
    information_card: {
      section_heading: "",
      section_text: "",
      cards: [
        {
          image_url: "",
          text_heading: "",
          text_description: ""
        }
      ]
    },
    technnology_used: {
      section_heading: "",
      section_text: "",
      sections_cards: [
        {
          tab_heading: "",
          related_technologies: [
            {
              icon_url: "",
              icon_title: "",
              icon_description: ""
            }
          ]
        }
      ]
    },
    our_experties_section: {
      section_heading: "",
      section_text: "",
      image: "",
      background_image_url: ""
    },
    about_us_section: {
      section_heading: "",
      section_text: ""
    }
  });

  // Pre-fill form data when in edit mode
  useEffect(() => {
    if (isEdit && initialData) {
      // Ensure all sections have proper structure even if they're null in the API response
      const processedData = {
        ...initialData,
        banner: initialData.banner || {
          banner_title_heading: "",
          banner_text: "",
          background_image_url: ""
        },
        services_tabs_section_image: initialData.services_tabs_section_image || {
          text: "",
          services_tabs_section_image: ""
        },
        stand_out_section: initialData.stand_out_section || {
          section_span: "",
          features: [
            {
              title: "",
              icon_url: ""
            }
          ]
        },
        information_card: initialData.information_card || {
          section_heading: "",
          section_text: "",
          cards: [
            {
              image_url: "",
              text_heading: "",
              text_description: ""
            }
          ]
        },
        technnology_used: initialData.technnology_used || {
          section_heading: "",
          section_text: "",
          sections_cards: [
            {
              tab_heading: "",
              related_technologies: [
                {
                  icon_url: "",
                  icon_title: "",
                  icon_description: ""
                }
              ]
            }
          ]
        },
        our_experties_section: initialData.our_experties_section || {
          section_heading: "",
          section_text: "",
          image: "",
          background_image_url: ""
        },
        about_us_section: JSON.parse(initialData.about_us_section) || {
          section_heading: "",
          section_text: ""
        }
      };
      setFormData(processedData);
    }
  }, [isEdit, initialData]);

  const { handleModalClose } = useModal()
  const handleSubmit = (event) => {
    event.preventDefault();
    console.log('Form Data:', formData);
    if (formData.page_heading) {
      if (isEdit) {
        console.log('initialData', initialData)
        console.log('formData', formData)

        dispatch(UpdateInternetMarkettingTemplatePage({ id: initialData.id, data: formData }))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Updated');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      } else {
        dispatch(SaveInternetMarketingTemplatePage(formData))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Created');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      }
    } else {
      toast.error('Page Heading is required');
    }
  };

  const handleInputChange = (section, subsection, index, field) => {
    if (section && subsection && index !== null && field) {
      // Handle nested array item field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? { ...item, [field]: event.target.value } : item
          )
        }
      }));
    } else if (section && subsection && index !== null) {
      // Handle array item (for simple arrays)
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? event.target.value : item
          )
        }
      }));
    } else if (section && subsection) {
      // Handle nested object field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: event.target.value
        }
      }));
    } else if (section) {
      // Handle direct field
      setFormData(prev => ({
        ...prev,
        [section]: event.target.value
      }));
    }
  };

  const handleSwitchChange = (field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.checked
    }));
  };

  const handleFileChange = async (section, subsection, index, field) => {
    const file = event.target.files[0];
    if (!file) return;

    const uploadKey = `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
    
    setUploadingFiles(prev => ({ ...prev, [uploadKey]: true }));
    setUploadErrors(prev => ({ ...prev, [uploadKey]: null }));

    try {
      const result = await dispatch(UploadServiceMediaAction({ file })).unwrap();
      
      if (result.success && result.data?.file_url) {
        if (section && subsection && index !== null && field) {
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? { ...item, [field]: result.data.file_url } : item
              )
            }
          }));
        } else if (section && subsection) {
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: result.data.file_url
            }
          }));
        } else if (section) {
          setFormData(prev => ({
            ...prev,
            [section]: result.data.file_url
          }));
        }
      }
    } catch (error) {
      console.error('File upload failed:', error);
      setUploadErrors(prev => ({ 
        ...prev, 
        [uploadKey]: error || 'File upload failed' 
      }));
    } finally {
      setUploadingFiles(prev => ({ ...prev, [uploadKey]: false }));
    }
  };

  const getUploadKey = (section, subsection, index, field) => {
    return `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
  };

  const isUploading = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadingFiles[key] || false;
  };

  const getUploadError = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadErrors[key] || null;
  };

  // Add/Remove functions for different sections
  const addStandOutFeature = () => {
    setFormData(prev => ({
      ...prev,
      stand_out_section: {
        ...prev.stand_out_section,
        features: [...prev.stand_out_section.features, { title: "", icon_url: "" }]
      }
    }));
  };

  const removeStandOutFeature = (index) => {
    setFormData(prev => ({
      ...prev,
      stand_out_section: {
        ...prev.stand_out_section,
        features: prev.stand_out_section.features.filter((_, i) => i !== index)
      }
    }));
  };

  const addInfoCard = () => {
    setFormData(prev => ({
      ...prev,
      information_card: {
        ...prev.information_card,
        cards: [...prev.information_card.cards, { image_url: "", text_heading: "", text_description: "" }]
      }
    }));
  };

  const removeInfoCard = (index) => {
    setFormData(prev => ({
      ...prev,
      information_card: {
        ...prev.information_card,
        cards: prev.information_card.cards.filter((_, i) => i !== index)
      }
    }));
  };

  const addTechnologySection = () => {
    setFormData(prev => ({
      ...prev,
      technnology_used: {
        ...prev.technnology_used,
        sections_cards: [...prev.technnology_used.sections_cards, { 
          tab_heading: "", 
          related_technologies: [{ icon_url: "", icon_title: "", icon_description: "" }] 
        }]
      }
    }));
  };

  const removeTechnologySection = (index) => {
    setFormData(prev => ({
      ...prev,
      technnology_used: {
        ...prev.technnology_used,
        sections_cards: prev.technnology_used.sections_cards.filter((_, i) => i !== index)
      }
    }));
  };

  const addRelatedTechnology = (sectionIndex) => {
    setFormData(prev => ({
      ...prev,
      technnology_used: {
        ...prev.technnology_used,
        sections_cards: prev.technnology_used.sections_cards.map((section, i) =>
          i === sectionIndex 
            ? { ...section, related_technologies: [...section.related_technologies, { icon_url: "", icon_title: "", icon_description: "" }] }
            : section
        )
      }
    }));
  };

  const removeRelatedTechnology = (sectionIndex, techIndex) => {
    setFormData(prev => ({
      ...prev,
      technnology_used: {
        ...prev.technnology_used,
        sections_cards: prev.technnology_used.sections_cards.map((section, i) =>
          i === sectionIndex 
            ? { ...section, related_technologies: section.related_technologies.filter((_, j) => j !== techIndex) }
            : section
        )
      }
    }));
  };



  return (
    <Container maxWidth="xl" sx={{ py: 4 }}>
      <Paper elevation={3} sx={{ p: 4 }}>
        <Box component="form" onSubmit={handleSubmit}>
          {/* Page Heading */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Page Heading
              </Typography>
              <TextField
                fullWidth
                label="Page Heading"
                value={formData.page_heading}
                onChange={(e) => handleInputChange('page_heading', null, null, null)}
                placeholder="Enter page heading"
                variant="outlined"
                margin="normal"
              />
            </CardContent>
          </Card>

          {/* Banner Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Banner Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Title Heading"
                    value={formData.banner.banner_title_heading}
                    onChange={(e) => handleInputChange('banner', 'banner_title_heading', null, null)}
                    placeholder="Enter banner title heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Text"
                    value={formData.banner.banner_text}
                    onChange={(e) => handleInputChange('banner', 'banner_text', null, null)}
                    placeholder="Enter banner text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('banner', 'background_image_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('banner', 'background_image_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('banner', 'background_image_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('banner', 'background_image_url', null, null)}</Alert>
                  )}
                  {formData.banner.background_image_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.banner.background_image_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Services Tabs Section Image */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Services Tabs Section Image
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Text"
                    value={formData.services_tabs_section_image.text}
                    onChange={(e) => handleInputChange('services_tabs_section_image', 'text', null, null)}
                    placeholder="Enter text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Section Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('services_tabs_section_image', 'services_tabs_section_image', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('services_tabs_section_image', 'services_tabs_section_image', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('services_tabs_section_image', 'services_tabs_section_image', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('services_tabs_section_image', 'services_tabs_section_image', null, null)}</Alert>
                  )}
                  {formData.services_tabs_section_image.services_tabs_section_image && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.services_tabs_section_image.services_tabs_section_image}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Stand Out Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Stand Out Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addStandOutFeature}
                >
                  Add Feature
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Span"
                value={formData.stand_out_section.section_span}
                onChange={(e) => handleInputChange('stand_out_section', 'section_span', null, null)}
                placeholder="Enter section span"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.stand_out_section.features.map((feature, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Feature {index + 1}</Typography>
                      {formData.stand_out_section.features.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeStandOutFeature(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={feature.title}
                          onChange={(e) => handleInputChange('stand_out_section', 'features', index, 'title')}
                          placeholder="Enter feature title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('stand_out_section', 'features', index, 'icon_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('stand_out_section', 'features', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('stand_out_section', 'features', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('stand_out_section', 'features', index, 'icon_url')}</Alert>
                        )}
                        {feature.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {feature.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Information Card Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Information Card Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addInfoCard}
                >
                  Add Info Card
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.information_card.section_heading}
                    onChange={(e) => handleInputChange('information_card', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.information_card.section_text}
                    onChange={(e) => handleInputChange('information_card', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.information_card.cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Info Card {index + 1}</Typography>
                      {formData.information_card.cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeInfoCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Image
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('information_card', 'cards', index, 'image_url')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('information_card', 'cards', index, 'image_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('information_card', 'cards', index, 'image_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('information_card', 'cards', index, 'image_url')}</Alert>
                        )}
                        {card.image_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.image_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Text Heading"
                          value={card.text_heading}
                          onChange={(e) => handleInputChange('information_card', 'cards', index, 'text_heading')}
                          placeholder="Enter text heading"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Text Description"
                          value={card.text_description}
                          onChange={(e) => handleInputChange('information_card', 'cards', index, 'text_description')}
                          placeholder="Enter text description"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Technology Used Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5">
                  Technology Used Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addTechnologySection}
                >
                  Add Technology Section
                </Button>
              </Box>
              <Grid container spacing={3} sx={{ mb: 3 }}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.technnology_used.section_heading}
                    onChange={(e) => handleInputChange('technnology_used', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.technnology_used.section_text}
                    onChange={(e) => handleInputChange('technnology_used', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
              </Grid>
              <Stack spacing={3}>
                {formData.technnology_used.sections_cards.map((section, sectionIndex) => (
                  <Card key={sectionIndex} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Technology Section {sectionIndex + 1}</Typography>
                      {formData.technnology_used.sections_cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeTechnologySection(sectionIndex)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <TextField
                      fullWidth
                      label="Tab Heading"
                      value={section.tab_heading}
                      onChange={(e) => handleInputChange('technnology_used', 'sections_cards', sectionIndex, 'tab_heading')}
                      placeholder="Enter tab heading"
                      variant="outlined"
                      margin="normal"
                      sx={{ mb: 3 }}
                    />
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="subtitle1">Related Technologies</Typography>
                      <Button
                        variant="outlined"
                        startIcon={<AddIcon />}
                        onClick={() => addRelatedTechnology(sectionIndex)}
                        size="small"
                      >
                        Add Technology
                      </Button>
                    </Box>
                    <Stack spacing={2}>
                      {section.related_technologies.map((tech, techIndex) => (
                        <Card key={techIndex} variant="outlined" sx={{ p: 2, bgcolor: 'white' }}>
                          <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                            <Typography variant="subtitle2">Technology {techIndex + 1}</Typography>
                            {section.related_technologies.length > 1 && (
                              <IconButton
                                color="error"
                                onClick={() => removeRelatedTechnology(sectionIndex, techIndex)}
                                size="small"
                              >
                                <DeleteIcon />
                              </IconButton>
                            )}
                          </Box>
                          <Grid container spacing={2}>
                            <Grid item xs={12}>
                              <Typography variant="subtitle2" gutterBottom>
                                Icon
                              </Typography>
                              <input
                                type="file"
                                accept="image/*"
                                onChange={(e) => handleFileChange('technnology_used', 'sections_cards', sectionIndex, 'related_technologies')}
                                style={{ width: '100%' }}
                              />
                              {isUploading('technnology_used', 'sections_cards', sectionIndex, 'related_technologies') && (
                                <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                                  <CircularProgress size={20} />
                                  <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                                </Box>
                              )}
                              {getUploadError('technnology_used', 'sections_cards', sectionIndex, 'related_technologies') && (
                                <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('technnology_used', 'sections_cards', sectionIndex, 'related_technologies')}</Alert>
                              )}
                              {tech.icon_url && (
                                <Alert severity="success" sx={{ mt: 1 }}>
                                  <Typography variant="body2">
                                    <strong>Uploaded:</strong> {tech.icon_url}
                                  </Typography>
                                </Alert>
                              )}
                            </Grid>
                            <Grid item xs={12} md={6}>
                              <TextField
                                fullWidth
                                label="Icon Title"
                                value={tech.icon_title}
                                onChange={(e) => handleInputChange('technnology_used', 'sections_cards', sectionIndex, 'related_technologies')}
                                placeholder="Enter icon title"
                                variant="outlined"
                                size="small"
                              />
                            </Grid>
                            <Grid item xs={12} md={6}>
                              <TextField
                                fullWidth
                                label="Icon Description"
                                value={tech.icon_description}
                                onChange={(e) => handleInputChange('technnology_used', 'sections_cards', sectionIndex, 'related_technologies')}
                                placeholder="Enter icon description"
                                variant="outlined"
                                size="small"
                              />
                            </Grid>
                          </Grid>
                        </Card>
                      ))}
                    </Stack>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Our Expertise Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                Our Expertise Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.our_experties_section.section_heading}
                    onChange={(e) => handleInputChange('our_experties_section', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.our_experties_section.section_text}
                    onChange={(e) => handleInputChange('our_experties_section', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={6}>
                  <Typography variant="subtitle1" gutterBottom>
                    Section Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('our_experties_section', 'image', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('our_experties_section', 'image', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('our_experties_section', 'image', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_experties_section', 'image', null, null)}</Alert>
                  )}
                  {formData.our_experties_section.image && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.our_experties_section.image}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
                <Grid item xs={12} md={6}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('our_experties_section', 'background_image_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('our_experties_section', 'background_image_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('our_experties_section', 'background_image_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_experties_section', 'background_image_url', null, null)}</Alert>
                  )}
                  {formData.our_experties_section.background_image_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.our_experties_section.background_image_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* About Us Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom>
                About Us Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.about_us_section.section_heading}
                    onChange={(e) => handleInputChange('about_us_section', 'section_heading', null, null)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.about_us_section.section_text}
                    onChange={(e) => handleInputChange('about_us_section', 'section_text', null, null)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Page Settings Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Settings
              </Typography>
              <Box sx={{ display: 'flex', gap: 4, flexWrap: 'wrap' }}>
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_review}
                      onChange={handleSwitchChange('enable_review')}
                      color="primary"
                    />
                  }
                  label="Enable Review on this page"
                />
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_faq}
                      onChange={handleSwitchChange('enable_faq')}
                      color="primary"
                    />
                  }
                  label="Enable FAQ on this page"
                />
              </Box>
            </CardContent>
          </Card>

          {/* Submit Button */}
          <Box display="flex" justifyContent="center" mt={4}>
            <Button
              type="submit"
              variant="contained"
              size="large"
              sx={{ py: 1, px: 4, fontSize: '1rem' }}
            >
              {isEdit ? 'Update Page' : 'Submit Form'}
            </Button>
          </Box>
        </Box>
      </Paper>
    </Container>
  );
};

export default InternetMarketingServiceForm;