import React, { useState, useEffect } from 'react';
import {
  Box,
  TextField,
  Button,
  Typography,
  Paper,
  Grid,
  Card,
  CardContent,
  IconButton,
  Divider,
  Container,
  Stack,
  CircularProgress,
  Alert,
  Switch,
  FormControlLabel
} from '@mui/material';
import { Add as AddIcon, Delete as DeleteIcon } from '@mui/icons-material';
import { useSelector, useDispatch } from 'react-redux';
import { GetAllServicesPages, SaveGraphicDesignTemplatePage, UpdateGraphicDesignTemplatePage, UploadServiceMediaAction } from '../../store/slices/servicesSlice/servicesSlice';
import { toast } from 'react-toastify';
import { GetAllServicesTabs } from '../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { useModal } from '../../hooks/useModal';

const GraphicDesignServiceForm = ({ isEdit, initialData, onSuccess, handleModalClose, setIsEdit }) => {
  const { handleModalClose: defaultHandleModalClose } = useModal()
  const { services } = useSelector(state => state.ServicesReducer);
  const dispatch = useDispatch();
  const [uploadingFiles, setUploadingFiles] = useState({});
  const [uploadErrors, setUploadErrors] = useState({});

  // Use passed handleModalClose if available, otherwise use default
  const modalCloseHandler = handleModalClose || defaultHandleModalClose;

  console.log('[services]', services);
  // console.log('Initial formData:', formData);

  const [formData, setFormData] = useState({
    page_heading: "",
    enable_review: false,
    enable_faq: false,
    banner: {
      banner_title_heading: "",
      banner_text: "",
      background_image_url: ""
    },
    stand_out_section: {
      section_span: "",
      section_text: "",
      background_file_url: ""
    },
    our_achievements_section: {
      features: [
        {
          title: "",
          text: ""
        }
      ]
    },
    our_services_section: {
      section_heading: "",
      section_text: "",
      section_background: "",
      services: [
        {
          title: "",
          sub_services: [
            {
              icon_url: "",
              title: "",
              section_span: "",
              description: ""
            }
          ]
        }
      ]
    },
    industries_we_serve_section: {
      section_heading: "",
      section_text: "",
      industries: [
        {
          image_url: "",
          title: ""
        }
      ]
    },
    our_top_clients_section: {
      section_heading: "",
      section_text: "",
      clients: [
        {
          image_url: "",
          name: "",
          description: "",
          position: ""
        }
      ]
    },
    portfolio_section: {
      section_span: "",
      section_heading: "",
      projects: [
        {
          image_url: "",
          title: "",
          description: ""
        }
      ]
    }
  });

  // Pre-fill form data when in edit mode
  useEffect(() => {
    if (isEdit && initialData) {
      // Ensure all sections have proper structure even if they're null in the API response
      const processedData = {
        ...initialData,
        banner: initialData.banner || {
          banner_title_heading: "",
          banner_text: "",
          background_image_url: ""
        },
        stand_out_section: initialData.stand_out_section || {
          section_span: "",
          section_text: "",
          background_file_url: ""
        },
        our_achievements_section: initialData.our_achievements_section || {
          features: [
            {
              title: "",
              text: ""
            }
          ]
        },
        our_services_section: initialData.our_services_section || {
          section_heading: "",
          section_text: "",
          section_background: "",
          services: [
            {
              title: "",
              sub_services: [
                {
                  icon_url: "",
                  title: "",
                  section_span: "",
                  description: ""
                }
              ]
            }
          ]
        },
        industries_we_serve_section: initialData.industries_we_serve_section || {
          section_heading: "",
          section_text: "",
          industries: [
            {
              image_url: "",
              title: ""
            }
          ]
        },
        our_top_clients_section: initialData.our_top_clients_section || {
          section_heading: "",
          section_text: "",
          clients: [
            {
              image_url: "",
              name: "",
              description: "",
              position: ""
            }
          ]
        },
        portfolio_section: initialData.portfolio_section || {
          section_span: "",
          section_heading: "",
          projects: [
            {
              image_url: "",
              title: "",
              description: ""
            }
          ]
        }
      };
      setFormData(processedData);
    }
  }, [isEdit, initialData]);

  // Debug: Log formData changes
  useEffect(() => {
    console.log('formData changed:', formData);
  }, [formData]);

  const handleInputChange = (section, subsection, index, field, subIndex, subField) => (event) => {
    const value = event.target.value;
    console.log('handleInputChange called:', { section, subsection, index, field, subIndex, subField, value });

    // For now, let's just handle the simple case to test
    if (section && !subsection && !index && !field && !subIndex && !subField) {
      // Handle direct field (like page_heading)
      console.log('Updating direct field:', { section, value });
      setFormData(prev => {
        const newData = { ...prev, [section]: value };
        console.log('New formData:', newData);
        return newData;
      });
    } else if (section && subsection && !index && !field && !subIndex && !subField) {
      // Handle nested object field (like banner.banner_title_heading)
      console.log('Updating nested object field:', { section, subsection, value });
      setFormData(prev => {
        const newData = {
          ...prev,
          [section]: {
            ...prev[section],
            [subsection]: value
          }
        };
        console.log('New formData:', newData);
        return newData;
      });
    } else {
      console.log('Complex case not implemented yet');
    }
  };

  const handleSwitchChange = (field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.checked
    }));
  };

  const handleFileChange = (section, subsection, index, field, subIndex, subField) => async (event) => {
    const file = event.target.files[0];
    if (!file) return;

    // Create a unique key for this file upload
    const uploadKey = `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${subIndex !== null ? subIndex : 'none'}_${subField || field || 'none'}`;

    // Set uploading state
    setUploadingFiles(prev => ({ ...prev, [uploadKey]: true }));
    setUploadErrors(prev => ({ ...prev, [uploadKey]: null }));

    try {
      const result = await dispatch(UploadServiceMediaAction({ file })).unwrap();

      if (result.success && result.data?.file_url) {
        // Update form data with the returned file_url
        if (section && subsection && index !== null && subIndex !== null && subField) {
          // Handle nested array within array (like sub_services within services)
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? {
                  ...item,
                  sub_services: item.sub_services.map((subItem, j) =>
                    j === subIndex ? { ...subItem, [subField]: result.data.file_url } : subItem
                  )
                } : item
              )
            }
          }));
        } else if (section && subsection && index !== null && field) {
          // Handle nested array item field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? { ...item, [field]: result.data.file_url } : item
              )
            }
          }));
        } else if (section && subsection) {
          // Handle nested object field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: result.data.file_url
            }
          }));
        } else if (section) {
          // Handle direct field
          setFormData(prev => ({
            ...prev,
            [section]: result.data.file_url
          }));
        }
      }
    } catch (error) {
      console.error('File upload failed:', error);
      setUploadErrors(prev => ({
        ...prev,
        [uploadKey]: error || 'File upload failed'
      }));
    } finally {
      setUploadingFiles(prev => ({ ...prev, [uploadKey]: false }));
    }
  };

  const getUploadKey = (section, subsection, index, field, subIndex, subField) => {
    return `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${subIndex !== null ? subIndex : 'none'}_${subField || field || 'none'}`;
  };

  const isUploading = (section, subsection, index, field, subIndex, subField) => {
    const key = getUploadKey(section, subsection, index, field, subIndex, subField);
    return uploadingFiles[key] || false;
  };

  const getUploadError = (section, subsection, index, field, subIndex, subField) => {
    const key = getUploadKey(section, subsection, index, field, subIndex, subField);
    return uploadErrors[key] || null;
  };

  const addAchievement = () => {
    setFormData(prev => ({
      ...prev,
      our_achievements_section: {
        ...prev.our_achievements_section,
        features: [...prev.our_achievements_section.features, { title: "", text: "" }]
      }
    }));
  };

  const removeAchievement = (index) => {
    setFormData(prev => ({
      ...prev,
      our_achievements_section: {
        ...prev.our_achievements_section,
        features: prev.our_achievements_section.features.filter((_, i) => i !== index)
      }
    }));
  };

  const addService = () => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services: [...prev.our_services_section.services, { title: "", sub_services: [{ icon_url: "", title: "", section_span: "", description: "" }] }]
      }
    }));
  };

  const removeService = (index) => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services: prev.our_services_section.services.filter((_, i) => i !== index)
      }
    }));
  };

  const addSubService = (serviceIndex) => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services: prev.our_services_section.services.map((service, i) =>
          i === serviceIndex
            ? {
              ...service,
              sub_services: [...service.sub_services, { icon_url: "", title: "", section_span: "", description: "" }]
            }
            : service
        )
      }
    }));
  };

  const removeSubService = (serviceIndex, subServiceIndex) => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services: prev.our_services_section.services.map((service, i) =>
          i === serviceIndex
            ? {
              ...service,
              sub_services: service.sub_services.filter((_, j) => j !== subServiceIndex)
            }
            : service
        )
      }
    }));
  };

  const addIndustry = () => {
    setFormData(prev => ({
      ...prev,
      industries_we_serve_section: {
        ...prev.industries_we_serve_section,
        industries: [...prev.industries_we_serve_section.industries, { image_url: "", title: "" }]
      }
    }));
  };

  const removeIndustry = (index) => {
    setFormData(prev => ({
      ...prev,
      industries_we_serve_section: {
        ...prev.industries_we_serve_section,
        industries: prev.industries_we_serve_section.industries.filter((_, i) => i !== index)
      }
    }));
  };

  const addClient = () => {
    setFormData(prev => ({
      ...prev,
      our_top_clients_section: {
        ...prev.our_top_clients_section,
        clients: [...prev.our_top_clients_section.clients, { image_url: "", name: "", description: "", position: "" }]
      }
    }));
  };

  const removeClient = (index) => {
    setFormData(prev => ({
      ...prev,
      our_top_clients_section: {
        ...prev.our_top_clients_section,
        clients: prev.our_top_clients_section.clients.filter((_, i) => i !== index)
      }
    }));
  };

  const addProject = () => {
    setFormData(prev => ({
      ...prev,
      portfolio_section: {
        ...prev.portfolio_section,
        projects: [...prev.portfolio_section.projects, { image_url: "", title: "", description: "" }]
      }
    }));
  };

  const removeProject = (index) => {
    setFormData(prev => ({
      ...prev,
      portfolio_section: {
        ...prev.portfolio_section,
        projects: prev.portfolio_section.projects.filter((_, i) => i !== index)
      }
    }));
  };

  const handleSubmit = (event) => {
    event.preventDefault();
    // console.log('Form Data:', formData);
    if (formData.page_heading) {
      if (isEdit) {

        console.log('initialData', initialData)
        console.log('formData', formData)


        dispatch(UpdateGraphicDesignTemplatePage({ id: initialData.id, data: formData }))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Updated');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      } else {
        dispatch(SaveGraphicDesignTemplatePage(formData))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Created');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)

            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      }
    } else {
      toast.error('Page Heading is required');
    }
  };

  return (
    <Container maxWidth="lg" sx={{ py: 4 }}>

      <Box component="form" onSubmit={handleSubmit}>
        {/* Page Heading */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Page Heading
            </Typography>
            <TextField
              fullWidth
              label="Page Heading"
              value={formData.page_heading}
              onChange={(e) => {
                console.log('Direct onChange called with value:', e.target.value);
                setFormData(prev => ({ ...prev, page_heading: e.target.value }));
              }}
              margin="normal"
            />
          </CardContent>
        </Card>

        {/* Banner Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Banner Section
            </Typography>
            <Grid container spacing={2}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Banner Title Heading"
                  value={formData.banner.banner_title_heading}
                  onChange={(e) => {
                    console.log('Banner title onChange called with value:', e.target.value);
                    setFormData(prev => ({
                      ...prev,
                      banner: {
                        ...prev.banner,
                        banner_title_heading: e.target.value
                      }
                    }));
                  }}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={6}>
                <TextField
                  fullWidth
                  label="Banner Text"
                  value={formData.banner.banner_text}
                  onChange={handleInputChange('banner', 'banner_text', null, null)}
                  margin="normal"
                  multiline
                  rows={3}
                />
              </Grid>
              <Grid item xs={12}>
                <Typography variant="subtitle2" gutterBottom>
                  Background Image
                </Typography>
                <input
                  type="file"
                  accept="image/*"
                  onChange={handleFileChange('banner', 'background_image_url', null, null)}
                  style={{ width: '100%' }}
                />
                {isUploading('banner', 'background_image_url', null, null) && (
                  <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                    <CircularProgress size={20} />
                    <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                  </Box>
                )}
                {getUploadError('banner', 'background_image_url', null, null) && (
                  <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('banner', 'background_image_url', null, null)}</Alert>
                )}
                {formData.banner.background_image_url && (
                  <Alert severity="success" sx={{ mt: 1 }}>
                    <Typography variant="body2">
                      <strong>Uploaded:</strong> {formData.banner.background_image_url}
                    </Typography>
                  </Alert>
                )}
              </Grid>
            </Grid>
          </CardContent>
        </Card>

        {/* Stand Out Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Stand Out Section
            </Typography>
            <Grid container spacing={2}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Span"
                  value={formData.stand_out_section.section_span}
                  onChange={handleInputChange('stand_out_section', 'section_span', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Text"
                  value={formData.stand_out_section.section_text}
                  onChange={handleInputChange('stand_out_section', 'section_text', null, null)}
                  margin="normal"
                  multiline
                  rows={3}
                />
              </Grid>
              <Grid item xs={12}>
                <Typography variant="subtitle2" gutterBottom>
                  Background File
                </Typography>
                <input
                  type="file"
                  accept="image/*"
                  onChange={handleFileChange('stand_out_section', 'background_file_url', null, null)}
                  style={{ width: '100%' }}
                />
                {isUploading('stand_out_section', 'background_file_url', null, null) && (
                  <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                    <CircularProgress size={20} />
                    <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                  </Box>
                )}
                {getUploadError('stand_out_section', 'background_file_url', null, null) && (
                  <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('stand_out_section', 'background_file_url', null, null)}</Alert>
                )}
                {formData.stand_out_section.background_file_url && (
                  <Alert severity="success" sx={{ mt: 1 }}>
                    <Typography variant="body2">
                      <strong>Uploaded:</strong> {formData.stand_out_section.background_file_url}
                    </Typography>
                  </Alert>
                )}
              </Grid>
            </Grid>
          </CardContent>
        </Card>

        {/* Our Achievements Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Our Achievements Section
            </Typography>
            <Stack spacing={3}>
              {formData.our_achievements_section.features.map((achievement, index) => (
                <Paper key={index} sx={{ p: 2 }}>
                  <Grid container spacing={2}>
                    <Grid item xs={12} md={5}>
                      <TextField
                        fullWidth
                        label="Achievement Title"
                        value={achievement.title}
                        onChange={handleInputChange('our_achievements_section', 'features', index, 'title')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={5}>
                      <TextField
                        fullWidth
                        label="Achievement Text"
                        value={achievement.text}
                        onChange={handleInputChange('our_achievements_section', 'features', index, 'text')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={2}>
                      <IconButton
                        onClick={() => removeAchievement(index)}
                        color="error"
                        disabled={formData.our_achievements_section.features.length === 1}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </Grid>
                  </Grid>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addAchievement}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Achievement
            </Button>
          </CardContent>
        </Card>

        {/* Our Services Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Our Services Section
            </Typography>
            <Grid container spacing={2} sx={{ mb: 3 }}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Heading"
                  value={formData.our_services_section.section_heading}
                  onChange={handleInputChange('our_services_section', 'section_heading', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Text"
                  value={formData.our_services_section.section_text}
                  onChange={handleInputChange('our_services_section', 'section_text', null, null)}
                  margin="normal"
                  multiline
                  rows={2}
                />
              </Grid>
              <Grid item xs={12}>
                <Typography variant="subtitle2" gutterBottom>
                  Section Background
                </Typography>
                <input
                  type="file"
                  accept="image/*"
                  onChange={handleFileChange('our_services_section', 'section_background', null, null)}
                  style={{ width: '100%' }}
                />
                {isUploading('our_services_section', 'section_background', null, null) && (
                  <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                    <CircularProgress size={20} />
                    <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                  </Box>
                )}
                {getUploadError('our_services_section', 'section_background', null, null) && (
                  <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_services_section', 'section_background', null, null)}</Alert>
                )}
                {formData.our_services_section.section_background && (
                  <Alert severity="success" sx={{ mt: 1 }}>
                    <Typography variant="body2">
                      <strong>Uploaded:</strong> {formData.our_services_section.section_background}
                    </Typography>
                  </Alert>
                )}
              </Grid>
            </Grid>
            <Stack spacing={3}>
              {formData.our_services_section.services.map((service, serviceIndex) => (
                <Paper key={serviceIndex} sx={{ p: 2 }}>
                  <Grid container spacing={2}>
                    <Grid item xs={12} md={11}>
                      <TextField
                        fullWidth
                        label="Service Title"
                        value={service.title}
                        onChange={handleInputChange('our_services_section', 'services', serviceIndex, 'title')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={1}>
                      <IconButton
                        onClick={() => removeService(serviceIndex)}
                        color="error"
                        disabled={formData.our_services_section.services.length === 1}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </Grid>
                    <Grid item xs={12}>
                      <Typography variant="subtitle1" gutterBottom>
                        Sub Services
                      </Typography>
                      <Stack spacing={2}>
                        {service.sub_services.map((subService, subServiceIndex) => (
                          <Paper key={subServiceIndex} sx={{ p: 2, backgroundColor: '#f5f5f5' }}>
                            <Grid container spacing={2}>

                              <Grid item xs={12} md={12}>
                                <TextField
                                  fullWidth
                                  label="Sub Service Title"
                                  value={subService.title}
                                  onChange={handleInputChange('our_services_section', 'services', serviceIndex, 'sub_services', subServiceIndex, 'title')}
                                  margin="normal"
                                />
                              </Grid>
                              <Grid item xs={12} md={12}>
                                <TextField
                                  fullWidth
                                  label="Section Span"
                                  value={subService.section_span}
                                  onChange={handleInputChange('our_services_section', 'services', serviceIndex, 'sub_services', subServiceIndex, 'section_span')}
                                  margin="normal"
                                />
                              </Grid>
                              <Grid item xs={12} md={12}>
                                <TextField
                                  fullWidth
                                  label="Description"
                                  value={subService.description}
                                  onChange={handleInputChange('our_services_section', 'services', serviceIndex, 'sub_services', subServiceIndex, 'description')}
                                  margin="normal"
                                  multiline
                                  rows={2}
                                />
                              </Grid>
                              <Grid item xs={12} md={12}>
                                <Typography variant="subtitle2" gutterBottom>
                                  Icon
                                </Typography>
                                <input
                                  type="file"
                                  accept="image/*"
                                  onChange={handleFileChange('our_services_section', 'services', serviceIndex, 'sub_services', subServiceIndex, 'icon_url')}
                                  style={{ width: '100%' }}
                                />
                                {isUploading('our_services_section', 'services', serviceIndex, 'sub_services', subServiceIndex, 'icon_url') && (
                                  <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                                    <CircularProgress size={20} />
                                    <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                                  </Box>
                                )}
                                {getUploadError('our_services_section', 'services', serviceIndex, 'sub_services', subServiceIndex, 'icon_url') && (
                                  <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_services_section', 'services', serviceIndex, 'sub_services', subServiceIndex, 'icon_url')}</Alert>
                                )}
                                {subService.icon_url && (
                                  <Alert severity="success" sx={{ mt: 1 }}>
                                    <Typography variant="body2">
                                      <strong>Uploaded:</strong> {subService.icon_url}
                                    </Typography>
                                  </Alert>
                                )}
                              </Grid>
                              <Grid item xs={12} md={12} style={{ display: 'flex', justifyContent: 'flex-end' }}>
                                <IconButton
                                  onClick={() => removeSubService(serviceIndex, subServiceIndex)}
                                  color="error"
                                  disabled={service.sub_services.length === 1}
                                >
                                  <DeleteIcon />
                                </IconButton>
                              </Grid>
                            </Grid>
                          </Paper>
                        ))}
                        <Button
                          startIcon={<AddIcon />}
                          onClick={() => addSubService(serviceIndex)}
                          variant="outlined"
                          size="small"
                          style={{ padding: '10px 20px' }}
                        >
                          Add Sub Service
                        </Button>
                      </Stack>
                    </Grid>
                  </Grid>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addService}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Service
            </Button>
          </CardContent>
        </Card>

        {/* Industries We Serve Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Industries We Serve Section
            </Typography>
            <Grid container spacing={2} sx={{ mb: 3 }}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Heading"
                  value={formData.industries_we_serve_section.section_heading}
                  onChange={handleInputChange('industries_we_serve_section', 'section_heading', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Text"
                  value={formData.industries_we_serve_section.section_text}
                  onChange={handleInputChange('industries_we_serve_section', 'section_text', null, null)}
                  margin="normal"
                  multiline
                  rows={2}
                />
              </Grid>
            </Grid>
            <Stack spacing={3}>
              {formData.industries_we_serve_section.industries.map((industry, index) => (
                <Paper key={index} sx={{ p: 2 }}>
                  <Grid container spacing={2}>

                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Industry Title"
                        value={industry.title}
                        onChange={handleInputChange('industries_we_serve_section', 'industries', index, 'title')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <Typography variant="subtitle2" gutterBottom>
                        Industry Image
                      </Typography>
                      <input
                        type="file"
                        accept="image/*"
                        onChange={handleFileChange('industries_we_serve_section', 'industries', index, 'image_url')}
                        style={{ width: '100%' }}
                      />
                      {isUploading('industries_we_serve_section', 'industries', index, 'image_url') && (
                        <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                          <CircularProgress size={20} />
                          <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                        </Box>
                      )}
                      {getUploadError('industries_we_serve_section', 'industries', index, 'image_url') && (
                        <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('industries_we_serve_section', 'industries', index, 'image_url')}</Alert>
                      )}
                      {industry.image_url && (
                        <Alert severity="success" sx={{ mt: 1 }}>
                          <Typography variant="body2">
                            <strong>Uploaded:</strong> {industry.image_url}
                          </Typography>
                        </Alert>
                      )}
                    </Grid>
                    <Grid item xs={12} md={12} style={{ display: 'flex', justifyContent: 'flex-end' }}>
                      <IconButton
                        onClick={() => removeIndustry(index)}
                        color="error"
                        disabled={formData.industries_we_serve_section.industries.length === 1}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </Grid>
                  </Grid>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addIndustry}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Industry
            </Button>
          </CardContent>
        </Card>

        {/* Our Top Clients Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Our Top Clients Section
            </Typography>
            <Grid container spacing={2} sx={{ mb: 3 }}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Heading"
                  value={formData.our_top_clients_section.section_heading}
                  onChange={handleInputChange('our_top_clients_section', 'section_heading', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Text"
                  value={formData.our_top_clients_section.section_text}
                  onChange={handleInputChange('our_top_clients_section', 'section_text', null, null)}
                  margin="normal"
                  multiline
                  rows={2}
                />
              </Grid>
            </Grid>
            <Stack spacing={3}>
              {formData.our_top_clients_section.clients.map((client, index) => (
                <Paper key={index} sx={{ p: 2 }}>
                  <Grid container spacing={2}>

                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Client Name"
                        value={client.name}
                        onChange={handleInputChange('our_top_clients_section', 'clients', index, 'name')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Description"
                        value={client.description}
                        onChange={handleInputChange('our_top_clients_section', 'clients', index, 'description')}
                        margin="normal"
                        multiline
                        rows={2}
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Position"
                        value={client.position}
                        onChange={handleInputChange('our_top_clients_section', 'clients', index, 'position')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <Typography variant="subtitle2" gutterBottom>
                        Client Image
                      </Typography>
                      <input
                        type="file"
                        accept="image/*"
                        onChange={handleFileChange('our_top_clients_section', 'clients', index, 'image_url')}
                        style={{ width: '100%' }}
                      />
                      {isUploading('our_top_clients_section', 'clients', index, 'image_url') && (
                        <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                          <CircularProgress size={20} />
                          <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                        </Box>
                      )}
                      {getUploadError('our_top_clients_section', 'clients', index, 'image_url') && (
                        <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_top_clients_section', 'clients', index, 'image_url')}</Alert>
                      )}
                      {client.image_url && (
                        <Alert severity="success" sx={{ mt: 1 }}>
                          <Typography variant="body2">
                            <strong>Uploaded:</strong> {client.image_url}
                          </Typography>
                        </Alert>
                      )}
                    </Grid>
                    <Grid item xs={12} md={12} style={{ display: 'flex', justifyContent: 'flex-end' }}>
                      <IconButton
                        onClick={() => removeClient(index)}
                        color="error"
                        disabled={formData.our_top_clients_section.clients.length === 1}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </Grid>
                  </Grid>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addClient}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Client
            </Button>
          </CardContent>
        </Card>

        {/* Portfolio Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Portfolio Section
            </Typography>
            <Grid container spacing={2} sx={{ mb: 3 }}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Span"
                  value={formData.portfolio_section.section_span}
                  onChange={handleInputChange('portfolio_section', 'section_span', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Heading"
                  value={formData.portfolio_section.section_heading}
                  onChange={handleInputChange('portfolio_section', 'section_heading', null, null)}
                  margin="normal"
                  multiline
                  rows={2}
                />
              </Grid>
            </Grid>
            <Stack spacing={3}>
              {formData.portfolio_section.projects.map((project, index) => (
                <Paper key={index} sx={{ p: 2 }}>
                  <Grid container spacing={2}>

                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Project Title"
                        value={project.title}
                        onChange={handleInputChange('portfolio_section', 'projects', index, 'title')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Description"
                        value={project.description}
                        onChange={handleInputChange('portfolio_section', 'projects', index, 'description')}
                        margin="normal"
                        multiline
                        rows={2}
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <Typography variant="subtitle2" gutterBottom>
                        Project Image
                      </Typography>
                      <input
                        type="file"
                        accept="image/*"
                        onChange={handleFileChange('portfolio_section', 'projects', index, 'image_url')}
                        style={{ width: '100%' }}
                      />
                      {isUploading('portfolio_section', 'projects', index, 'image_url') && (
                        <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                          <CircularProgress size={20} />
                          <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                        </Box>
                      )}
                      {getUploadError('portfolio_section', 'projects', index, 'image_url') && (
                        <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('portfolio_section', 'projects', index, 'image_url')}</Alert>
                      )}
                      {project.image_url && (
                        <Alert severity="success" sx={{ mt: 1 }}>
                          <Typography variant="body2">
                            <strong>Uploaded:</strong> {project.image_url}
                          </Typography>
                        </Alert>
                      )}
                    </Grid>
                    <Grid item xs={12} md={12} style={{ display: 'flex', justifyContent: 'flex-end' }}>
                      <IconButton
                        onClick={() => removeProject(index)}
                        color="error"
                        disabled={formData.portfolio_section.projects.length === 1}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </Grid>
                  </Grid>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addProject}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Project
            </Button>
          </CardContent>
        </Card>

        {/* Page Settings Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Page Settings
            </Typography>
            <Box sx={{ display: 'flex', gap: 4, flexWrap: 'wrap' }}>
              <FormControlLabel
                control={
                  <Switch
                    checked={formData.enable_review}
                    onChange={handleSwitchChange('enable_review')}
                    color="primary"
                  />
                }
                label="Enable Review on this page"
              />
              <FormControlLabel
                control={
                  <Switch
                    checked={formData.enable_faq}
                    onChange={handleSwitchChange('enable_faq')}
                    color="primary"
                  />
                }
                label="Enable FAQ on this page"
              />
            </Box>
          </CardContent>
        </Card>

        {/* Submit Button */}
        <Box sx={{ display: 'flex', justifyContent: 'center', mt: 4 }}>
          <Button
            type="submit"
            variant="contained"
            size="large"
            sx={{ py: 1, px: 4, fontSize: '1rem' }}
          >
            {isEdit ? 'Update Page' : 'Submit Form'}
          </Button>
        </Box>
      </Box>
    </Container>
  );
};

export default GraphicDesignServiceForm;