import React, { useState, useEffect } from 'react';
import {
  Box,
  TextField,
  Button,
  Typography,
  Paper,
  Grid,
  Card,
  CardContent,
  IconButton,
  Divider,
  Container,
  Stack,
  CircularProgress,
  Alert,
  Switch,
  FormControlLabel
} from '@mui/material';
import { Add as AddIcon, Delete as DeleteIcon } from '@mui/icons-material';
import { useSelector, useDispatch } from 'react-redux';
import { GetAllServicesPages, SaveDigitalMarketingTemplatePage, UploadServiceMediaAction, UpdateDigitalMarkettingTemplatePage } from '../../store/slices/servicesSlice/servicesSlice';
import { toast } from 'react-toastify';
import { useModal } from '../../hooks/useModal';

const DigitalMarkettingServiceForm = ({ isEdit, initialData, onSuccess, handleModalClose, setIsEdit }) => {
  const { handleModalClose: defaultHandleModalClose } = useModal()
  const { services } = useSelector(state => state.ServicesReducer);
  const dispatch = useDispatch();
  const [uploadingFiles, setUploadingFiles] = useState({});
  const [uploadErrors, setUploadErrors] = useState({});

  // Use passed handleModalClose if available, otherwise use default
  const modalCloseHandler = handleModalClose || defaultHandleModalClose;

  console.log('[services]', services);

  const [formData, setFormData] = useState({
    page_heading: "",
    enable_review: false,
    enable_faq: false,
    banner: {
      banner_title_heading: "",
      banner_text: "",
      background_image_url: ""
    },
    service_section_Tabs: {
      tabs: [
        {
          heading: "",
          text: "",
          image: ""
        }
      ]
    },
    services_cards: {
      section_heading: "",
      section_text: "",
      cards: [
        {
          image: "",
          heading: "",
          content: ""
        }
      ]
    },
    all_services_grid: {
      heading: "",
      services: [""]
    },
    technologies_we_work_with: {
      section_heading: "",
      service_tab: [
        {
          section_heading: "",
          features: [
            {
              title: "",
              text: "",
              image: ""
            }
          ]
        }
      ]
    },
    info_cards_section: {
      section_heading: "",
      cards: [
        {
          image: "",
          heading: "",
          content: ""
        }
      ]
    }
  });

  // Pre-fill form data when in edit mode
  useEffect(() => {
    if (isEdit && initialData) {
      // Ensure all sections have proper structure even if they're null in the API response
      const processedData = {
        ...initialData,
        banner: initialData.banner || {
          banner_title_heading: "",
          banner_text: "",
          background_image_url: ""
        },
        service_section_Tabs: initialData.service_section_Tabs || {
          tabs: [
            {
              heading: "",
              text: "",
              image: ""
            }
          ]
        },
        services_cards: initialData.services_cards || {
          section_heading: "",
          section_text: "",
          cards: [
            {
              image: "",
              heading: "",
              content: ""
            }
          ]
        },
        all_services_grid: initialData.all_services_grid || {
          heading: "",
          services: [""]
        },
        technologies_we_work_with: initialData.technologies_we_work_with || {
          section_heading: "",
          service_tab: [
            {
              section_heading: "",
              features: [
                {
                  title: "",
                  text: "",
                  image: ""
                }
              ]
            }
          ]
        },
        info_cards_section: initialData.info_cards_section || {
          section_heading: "",
          cards: [
            {
              image: "",
              heading: "",
              content: ""
            }
          ]
        }
      };
      setFormData(processedData);
    }
  }, [isEdit, initialData]);

  const handleInputChange = (section, subsection, index, field) => {
    if (section && subsection && index !== null && field) {
      // Handle nested array item field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? { ...item, [field]: event.target.value } : item
          )
        }
      }));
    } else if (section && subsection && index !== null) {
      // Handle array item (for services array)
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? event.target.value : item
          )
        }
      }));
    } else if (section && subsection) {
      // Handle nested object field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: event.target.value
        }
      }));
    } else if (section) {
      // Handle direct field
      setFormData(prev => ({
        ...prev,
        [section]: event.target.value
      }));
    }
  };

  const handleFileChange = async (section, subsection, index, field) => {
    const file = event.target.files[0];
    if (!file) return;

    // Create a unique key for this file upload
    const uploadKey = `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
    
    // Set uploading state
    setUploadingFiles(prev => ({ ...prev, [uploadKey]: true }));
    setUploadErrors(prev => ({ ...prev, [uploadKey]: null }));

    try {
      const result = await dispatch(UploadServiceMediaAction({ file })).unwrap();
      
      if (result.success && result.data?.file_url) {
        // Update form data with the returned file_url
        if (section && subsection && index !== null && field) {
          // Handle nested array item field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? { ...item, [field]: result.data.file_url } : item
              )
            }
          }));
        } else if (section && subsection) {
          // Handle nested object field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: result.data.file_url
            }
          }));
        } else if (section) {
          // Handle direct field
          setFormData(prev => ({
            ...prev,
            [section]: result.data.file_url
          }));
        }
      }
    } catch (error) {
      console.error('File upload failed:', error);
      setUploadErrors(prev => ({ 
        ...prev, 
        [uploadKey]: error || 'File upload failed' 
      }));
    } finally {
      setUploadingFiles(prev => ({ ...prev, [uploadKey]: false }));
    }
  };

  const getUploadKey = (section, subsection, index, field) => {
    return `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
  };

  const isUploading = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadingFiles[key] || false;
  };

  const getUploadError = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadErrors[key] || null;
  };

  const addTab = () => {
    setFormData(prev => ({
      ...prev,
      service_section_Tabs: {
        ...prev.service_section_Tabs,
        tabs: [...prev.service_section_Tabs.tabs, { heading: "", text: "", image: "" }]
      }
    }));
  };

  const removeTab = (index) => {
    setFormData(prev => ({
      ...prev,
      service_section_Tabs: {
        ...prev.service_section_Tabs,
        tabs: prev.service_section_Tabs.tabs.filter((_, i) => i !== index)
      }
    }));
  };

  const addServiceCard = () => {
    setFormData(prev => ({
      ...prev,
      services_cards: {
        ...prev.services_cards,
        cards: [...prev.services_cards.cards, { image: "", heading: "", content: "" }]
      }
    }));
  };

  const removeServiceCard = (index) => {
    setFormData(prev => ({
      ...prev,
      services_cards: {
        ...prev.services_cards,
        cards: prev.services_cards.cards.filter((_, i) => i !== index)
      }
    }));
  };

  const addService = () => {
    setFormData(prev => ({
      ...prev,
      all_services_grid: {
        ...prev.all_services_grid,
        services: [...prev.all_services_grid.services, ""]
      }
    }));
  };

  const removeService = (index) => {
    setFormData(prev => ({
      ...prev,
      all_services_grid: {
        ...prev.all_services_grid,
        services: prev.all_services_grid.services.filter((_, i) => i !== index)
      }
    }));
  };

  const addServiceTab = () => {
    setFormData(prev => ({
      ...prev,
      technologies_we_work_with: {
        ...prev.technologies_we_work_with,
        service_tab: [...prev.technologies_we_work_with.service_tab, { 
          section_heading: "", 
          features: [{ title: "", text: "", image: "" }] 
        }]
      }
    }));
  };

  const removeServiceTab = (index) => {
    setFormData(prev => ({
      ...prev,
      technologies_we_work_with: {
        ...prev.technologies_we_work_with,
        service_tab: prev.technologies_we_work_with.service_tab.filter((_, i) => i !== index)
      }
    }));
  };

  const addFeature = (tabIndex) => {
    setFormData(prev => ({
      ...prev,
      technologies_we_work_with: {
        ...prev.technologies_we_work_with,
        service_tab: prev.technologies_we_work_with.service_tab.map((tab, i) =>
          i === tabIndex 
            ? { ...tab, features: [...tab.features, { title: "", text: "", image: "" }] }
            : tab
        )
      }
    }));
  };

  const removeFeature = (tabIndex, featureIndex) => {
    setFormData(prev => ({
      ...prev,
      technologies_we_work_with: {
        ...prev.technologies_we_work_with,
        service_tab: prev.technologies_we_work_with.service_tab.map((tab, i) =>
          i === tabIndex 
            ? { ...tab, features: tab.features.filter((_, fi) => fi !== featureIndex) }
            : tab
        )
      }
    }));
  };

  const addInfoCard = () => {
    setFormData(prev => ({
      ...prev,
      info_cards_section: {
        ...prev.info_cards_section,
        cards: [...prev.info_cards_section.cards, { image: "", heading: "", content: "" }]
      }
    }));
  };

  const removeInfoCard = (index) => {
    setFormData(prev => ({
      ...prev,
      info_cards_section: {
        ...prev.info_cards_section,
        cards: prev.info_cards_section.cards.filter((_, i) => i !== index)
      }
    }));
  };

  const handleSwitchChange = (field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.checked
    }));
  };

  const handleSubmit = (event) => {
    event.preventDefault();
    console.log('Form Data:', formData);
    if (formData.page_heading) {
      if (isEdit) {
        console.log('initialData', initialData)
        console.log('formData', formData)

        dispatch(UpdateDigitalMarkettingTemplatePage({ id: initialData.id, data: formData }))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Updated');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      } else {
        dispatch(SaveDigitalMarketingTemplatePage(formData))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Created');
              modalCloseHandler()
              dispatch(GetAllServicesPages())
              setIsEdit(false)
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          })
      }
    } else {
      toast.error('Page Heading is required');
    }
  };

  return (
    <Container maxWidth="xl" sx={{ py: 4 }}>
      <Paper elevation={3} sx={{ p: 4 }}>

        <Box component="form" onSubmit={handleSubmit}>
          {/* Page Heading */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Heading
              </Typography>
              <TextField
                fullWidth
                label="Page Heading"
                value={formData.page_heading}
                onChange={(e) => handleInputChange('page_heading', null, null, null)}
                placeholder="Enter page heading"
                variant="outlined"
                margin="normal"
              />
            </CardContent>
          </Card>

          {/* Banner Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Banner Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Title Heading"
                    value={formData.banner.banner_title_heading}
                    onChange={(e) => handleInputChange('banner', 'banner_title_heading', null, null)}
                    placeholder="Enter banner title heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Text"
                    value={formData.banner.banner_text}
                    onChange={(e) => handleInputChange('banner', 'banner_text', null, null)}
                    placeholder="Enter banner text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('banner', 'background_image_url', null, null)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('banner', 'background_image_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('banner', 'background_image_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('banner', 'background_image_url', null, null)}</Alert>
                  )}
                  {formData.banner.background_image_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.banner.background_image_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Service Section Tabs */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Service Section Tabs
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addTab}
                >
                  Add Tab
                </Button>
              </Box>
              <Stack spacing={3}>
                {formData.service_section_Tabs.tabs.map((tab, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Tab {index + 1}</Typography>
                      {formData.service_section_Tabs.tabs.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeTab(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Heading"
                          value={tab.heading}
                          onChange={(e) => handleInputChange('service_section_Tabs', 'tabs', index, 'heading')}
                          placeholder="Enter tab heading"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Text"
                          value={tab.text}
                          onChange={(e) => handleInputChange('service_section_Tabs', 'tabs', index, 'text')}
                          placeholder="Enter tab text"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Tab Image
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('service_section_Tabs', 'tabs', index, 'image')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('service_section_Tabs', 'tabs', index, 'image') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('service_section_Tabs', 'tabs', index, 'image') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('service_section_Tabs', 'tabs', index, 'image')}</Alert>
                        )}
                        {tab.image && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {tab.image}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Services Cards Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Services Cards Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addServiceCard}
                >
                  Add Card
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.services_cards.section_heading}
                onChange={(e) => handleInputChange('services_cards', 'section_heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Text"
                value={formData.services_cards.section_text}
                onChange={(e) => handleInputChange('services_cards', 'section_text', null, null)}
                placeholder="Enter section text"
                variant="outlined"
                multiline
                rows={3}
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.services_cards.cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Card {index + 1}</Typography>
                      {formData.services_cards.cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeServiceCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Card Image
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('services_cards', 'cards', index, 'image')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('services_cards', 'cards', index, 'image') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('services_cards', 'cards', index, 'image') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('services_cards', 'cards', index, 'image')}</Alert>
                        )}
                        {card.image && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.image}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12}>
                        <TextField
                          fullWidth
                          label="Heading"
                          value={card.heading}
                          onChange={(e) => handleInputChange('services_cards', 'cards', index, 'heading')}
                          placeholder="Enter card heading"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12}>
                        <TextField
                          fullWidth
                          label="Content"
                          value={card.content}
                          onChange={(e) => handleInputChange('services_cards', 'cards', index, 'content')}
                          placeholder="Enter card content"
                          variant="outlined"
                          multiline
                          rows={4}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* All Services Grid */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  All Services Grid
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addService}
                >
                  Add Service
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.all_services_grid.heading}
                onChange={(e) => handleInputChange('all_services_grid', 'heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={2}>
                {formData.all_services_grid.services.map((service, index) => (
                  <Box key={index} display="flex" gap={2} alignItems="center">
                    <TextField
                      fullWidth
                      label={`Service ${index + 1}`}
                      value={service}
                      onChange={(e) => handleInputChange('all_services_grid', 'services', index, null)}
                      placeholder="Enter service name"
                      variant="outlined"
                      size="small"
                    />
                    {formData.all_services_grid.services.length > 1 && (
                      <IconButton
                        color="error"
                        onClick={() => removeService(index)}
                        size="small"
                      >
                        <DeleteIcon />
                      </IconButton>
                    )}
                  </Box>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Technologies We Work With Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Technologies We Work With
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addServiceTab}
                >
                  Add Service Tab
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.technologies_we_work_with.section_heading}
                onChange={(e) => handleInputChange('technologies_we_work_with', 'section_heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.technologies_we_work_with.service_tab.map((tab, tabIndex) => (
                  <Card key={tabIndex} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Service Tab {tabIndex + 1}</Typography>
                      {formData.technologies_we_work_with.service_tab.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeServiceTab(tabIndex)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <TextField
                      fullWidth
                      label="Tab Section Heading"
                      value={tab.section_heading}
                      onChange={(e) => handleInputChange('technologies_we_work_with', 'service_tab', tabIndex, 'section_heading')}
                      placeholder="Enter tab section heading"
                      variant="outlined"
                      size="small"
                      sx={{ mb: 2 }}
                    />
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="subtitle1">Features</Typography>
                      <Button
                        variant="outlined"
                        startIcon={<AddIcon />}
                        onClick={() => addFeature(tabIndex)}
                        size="small"
                      >
                        Add Feature
                      </Button>
                    </Box>
                    <Stack spacing={2}>
                      {tab.features.map((feature, featureIndex) => (
                        <Card key={featureIndex} variant="outlined" sx={{ p: 2, bgcolor: 'white' }}>
                          <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                            <Typography variant="subtitle2">Feature {featureIndex + 1}</Typography>
                            {tab.features.length > 1 && (
                              <IconButton
                                color="error"
                                onClick={() => removeFeature(tabIndex, featureIndex)}
                                size="small"
                              >
                                <DeleteIcon />
                              </IconButton>
                            )}
                          </Box>
                          <Grid container spacing={2}>
                            <Grid item xs={12}>
                              <TextField
                                fullWidth
                                label="Title"
                                value={feature.title}
                                onChange={(e) => handleInputChange('technologies_we_work_with', 'service_tab', tabIndex, 'features')}
                                placeholder="Enter feature title"
                                variant="outlined"
                                size="small"
                              />
                            </Grid>
                            <Grid item xs={12}>
                              <TextField
                                fullWidth
                                label="Text"
                                value={feature.text}
                                onChange={(e) => handleInputChange('technologies_we_work_with', 'service_tab', tabIndex, 'features')}
                                placeholder="Enter feature text"
                                variant="outlined"
                                multiline
                                rows={3}
                                size="small"
                              />
                            </Grid>
                            <Grid item xs={12}>
                              <Typography variant="subtitle2" gutterBottom>
                                Feature Image
                              </Typography>
                              <input
                                type="file"
                                accept="image/*"
                                onChange={(e) => handleFileChange('technologies_we_work_with', 'service_tab', tabIndex, 'features')}
                                style={{ width: '100%' }}
                              />
                              {isUploading('technologies_we_work_with', 'service_tab', tabIndex, 'features') && (
                                <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                                  <CircularProgress size={20} />
                                  <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                                </Box>
                              )}
                              {getUploadError('technologies_we_work_with', 'service_tab', tabIndex, 'features') && (
                                <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('technologies_we_work_with', 'service_tab', tabIndex, 'features')}</Alert>
                              )}
                              {feature.image && (
                                <Alert severity="success" sx={{ mt: 1 }}>
                                  <Typography variant="body2">
                                    <strong>Uploaded:</strong> {feature.image}
                                  </Typography>
                                </Alert>
                              )}
                            </Grid>
                          </Grid>
                        </Card>
                      ))}
                    </Stack>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Info Cards Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Info Cards Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addInfoCard}
                >
                  Add Card
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.info_cards_section.section_heading}
                onChange={(e) => handleInputChange('info_cards_section', 'section_heading', null, null)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.info_cards_section.cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Card {index + 1}</Typography>
                      {formData.info_cards_section.cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeInfoCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Card Image
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('info_cards_section', 'cards', index, 'image')}
                          style={{ width: '100%' }}
                        />
                        {isUploading('info_cards_section', 'cards', index, 'image') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('info_cards_section', 'cards', index, 'image') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('info_cards_section', 'cards', index, 'image')}</Alert>
                        )}
                        {card.image && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {card.image}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12}>
                        <TextField
                          fullWidth
                          label="Heading"
                          value={card.heading}
                          onChange={(e) => handleInputChange('info_cards_section', 'cards', index, 'heading')}
                          placeholder="Enter card heading"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12}>
                        <TextField
                          fullWidth
                          label="Content"
                          value={card.content}
                          onChange={(e) => handleInputChange('info_cards_section', 'cards', index, 'content')}
                          placeholder="Enter card content"
                          variant="outlined"
                          multiline
                          rows={4}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Page Settings Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Settings
              </Typography>
              <Box sx={{ display: 'flex', gap: 4, flexWrap: 'wrap' }}>
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_review}
                      onChange={handleSwitchChange('enable_review')}
                      color="primary"
                    />
                  }
                  label="Enable Review on this page"
                />
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_faq}
                      onChange={handleSwitchChange('enable_faq')}
                      color="primary"
                    />
                  }
                  label="Enable FAQ on this page"
                />
              </Box>
            </CardContent>
          </Card>

          {/* Submit Button */}
          <Box display="flex" justifyContent="center" mt={4}>
            <Button
              type="submit"
              variant="contained"
              size="large"
              sx={{ py: 1, px: 4, fontSize: '1rem' }}
            >
              {isEdit ? 'Update Page' : 'Submit Form'}
            </Button>
          </Box>
        </Box>
      </Paper>
    </Container>
  );
};

export default DigitalMarkettingServiceForm;