import React, { useState, useEffect } from 'react';
import {
  Box,
  TextField,
  Button,
  Typography,
  Paper,
  Grid,
  Card,
  CardContent,
  IconButton,
  Divider,
  Container,
  Stack,
  CircularProgress,
  Alert,
  Switch,
  FormControlLabel
} from '@mui/material';
import { Add as AddIcon, Delete as DeleteIcon } from '@mui/icons-material';
import { useSelector, useDispatch } from 'react-redux';
import { GetAllServicesPages, SaveBrandAgencyTemplatePage, UpdateBrandAgencyTemplatePage, UploadServiceMediaAction } from '../../store/slices/servicesSlice/servicesSlice';
import { toast } from 'react-toastify';
import { useModal } from '../../hooks/useModal';
import { GetAllServicesTabs } from '../../store/slices/propertyManagementSlice/propertyManagementSlice';

const BrandAgencyServiceForm = ({ isEdit, initialData, onSuccess, handleModalClose, setIsEdit }) => {
  const { services } = useSelector(state => state.ServicesReducer);
  const dispatch = useDispatch();
  const [uploadingFiles, setUploadingFiles] = useState({});
  const [uploadErrors, setUploadErrors] = useState({});
  const { handleModalClose: defaultHandleModalClose } = useModal()

  // Use passed handleModalClose if available, otherwise use default
  const modalCloseHandler = handleModalClose || defaultHandleModalClose;

  console.log('[services]', services);

  const [formData, setFormData] = useState({
    page_heading: "",
    enable_review: false,
    enable_faq: false,
    banner: {
      banner_title_heading: "",
      banner_text: "",
      background_image_url: ""
    },
    services_tabs_section_image: "",
    stand_out_section: {
      section_heading: "",
      section_text: ""
    },
    why_us_image: "",
    why_choose_us_section: {
      section_heading: "",
      features: [
        {
          title: "",
          text: "",
          image: ""
        }
      ]
    },
    all_services_grid: {
      section_heading: "",
      services: [""]
    },
    our_services_section: {
      section_heading: "",
      section_background: "",
      services: [
        {
          title: "",
          description: "",
          image: "",
          buttton_text: "",
          button_link: ""
        }
      ]
    }
  });

  // Pre-fill form data when in edit mode
  useEffect(() => {
    if (isEdit && initialData) {
      console.log('initialData', initialData);
      // Ensure all sections have proper structure even if they're null in the API response
      const processedData = {
        page_heading: initialData.page_heading || "",
        enable_review: initialData.enable_review || false,
        enable_faq: initialData.enable_faq || false,
        banner: {
          banner_title_heading: initialData.banner?.banner_title_heading || "",
          banner_text: initialData.banner?.banner_text || "",
          background_image_url: initialData.banner?.background_image_url || ""
        },
        services_tabs_section_image: initialData.services_tabs_section_image || "",
        stand_out_section: {
          section_heading: initialData.stand_out_section?.section_heading || "",
          section_text: initialData.stand_out_section?.section_text || ""
        },
        why_us_image: initialData.why_us_image || "",
        why_choose_us_section: {
          section_heading: initialData.why_choose_us_section?.section_heading || "",
          features: initialData.why_choose_us_section?.features || [
            {
              title: "",
              text: "",
              image: ""
            }
          ]
        },
        all_services_grid: {
          section_heading: initialData.all_services_grid?.section_heading || "",
          services: initialData.all_services_grid?.services || [""]
        },
        our_services_section: {
          section_heading: initialData.our_services_section?.section_heading || "",
          section_background: initialData.our_services_section?.section_background || "",
          services: initialData.our_services_section?.services || [
            {
              title: "",
              description: "",
              image: "",
              buttton_text: "",
              button_link: ""
            }
          ]
        }
      };
      setFormData(processedData);
    }
  }, [isEdit, initialData]);

  // Debug: Log formData changes
  useEffect(() => {
    console.log('formData changed:', formData);
  }, [formData]);

  const handleInputChange = (section, subsection, index, field) => (event) => {
    const value = event.target.value;
    
    if (section && subsection && index !== null && field) {
      // Handle nested array item field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? { ...item, [field]: value } : item
          )
        }
      }));
    } else if (section && subsection && index !== null) {
      // Handle array item (for services array)
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? value : item
          )
        }
      }));
    } else if (section && subsection) {
      // Handle nested object field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: value
        }
      }));
    } else if (section) {
      // Handle direct field
      setFormData(prev => ({
        ...prev,
        [section]: value
      }));
    }
  };

  const handleSwitchChange = (field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.checked
    }));
  };

  const handleFileChange = (section, subsection, index, field) => async (event) => {
    const file = event.target.files[0];
    if (!file) return;

    // Create a unique key for this file upload
    const uploadKey = `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;

    // Set uploading state
    setUploadingFiles(prev => ({ ...prev, [uploadKey]: true }));
    setUploadErrors(prev => ({ ...prev, [uploadKey]: null }));

    try {
      const result = await dispatch(UploadServiceMediaAction({ file })).unwrap();

      if (result.success && result.data?.file_url) {
        // Update form data with the returned file_url
        if (section && subsection && index !== null && field) {
          // Handle nested array item field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? { ...item, [field]: result.data.file_url } : item
              )
            }
          }));
        } else if (section && subsection) {
          // Handle nested object field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: result.data.file_url
            }
          }));
        } else if (section) {
          // Handle direct field
          setFormData(prev => ({
            ...prev,
            [section]: result.data.file_url
          }));
        }
      }
    } catch (error) {
      console.error('File upload failed:', error);
      setUploadErrors(prev => ({
        ...prev,
        [uploadKey]: error || 'File upload failed'
      }));
    } finally {
      setUploadingFiles(prev => ({ ...prev, [uploadKey]: false }));
    }
  };

  const getUploadKey = (section, subsection, index, field) => {
    return `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
  };

  const isUploading = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadingFiles[key] || false;
  };

  const getUploadError = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadErrors[key] || null;
  };

  const addFeature = () => {
    setFormData(prev => ({
      ...prev,
      why_choose_us_section: {
        ...prev.why_choose_us_section,
        features: [...prev.why_choose_us_section.features, { title: "", text: "", image: "" }]
      }
    }));
  };

  const removeFeature = (index) => {
    setFormData(prev => ({
      ...prev,
      why_choose_us_section: {
        ...prev.why_choose_us_section,
        features: prev.why_choose_us_section.features.filter((_, i) => i !== index)
      }
    }));
  };

  const addService = () => {
    setFormData(prev => ({
      ...prev,
      all_services_grid: {
        ...prev.all_services_grid,
        services: [...prev.all_services_grid.services, ""]
      }
    }));
  };

  const removeService = (index) => {
    setFormData(prev => ({
      ...prev,
      all_services_grid: {
        ...prev.all_services_grid,
        services: prev.all_services_grid.services.filter((_, i) => i !== index)
      }
    }));
  };

  const addOurService = () => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services: [...prev.our_services_section.services, { title: "", description: "", image: "", buttton_text: "", button_link: "" }]
      }
    }));
  };

  const removeOurService = (index) => {
    setFormData(prev => ({
      ...prev,
      our_services_section: {
        ...prev.our_services_section,
        services: prev.our_services_section.services.filter((_, i) => i !== index)
      }
    }));
  };

  const handleSubmit = (event) => {
    event.preventDefault();
    console.log('Form Data:', formData);
    if (formData.page_heading) {
      if (isEdit) {
        console.log('initialData', initialData);
        console.log('formData', formData);
        
        dispatch(UpdateBrandAgencyTemplatePage({ id: initialData.id, data: formData }))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Updated');
              modalCloseHandler();
              dispatch(GetAllServicesPages());
              if (onSuccess) onSuccess(response.data);
              if (setIsEdit) setIsEdit(false);
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          });
      } else {
        dispatch(SaveBrandAgencyTemplatePage(formData))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Created');
              modalCloseHandler();
              dispatch(GetAllServicesPages());
              if (onSuccess) onSuccess(response.data);
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          });
      }
    } else {
      toast.error('Page Heading is required');
    }
  };

  return (
    <Container maxWidth="lg" sx={{ py: 4 }}>
      {/* Safety check - ensure formData is properly initialized */}
      {!formData || !formData.why_choose_us_section ? (
        <Box sx={{ display: 'flex', justifyContent: 'center', py: 4 }}>
          <CircularProgress />
        </Box>
      ) : (
        <Box component="form" onSubmit={handleSubmit}>
        {/* Page Heading */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Page Heading
            </Typography>
            <TextField
              fullWidth
              label="Page Heading"
              value={formData.page_heading}
              onChange={(e) => handleInputChange('page_heading', null, null, null)(e)}
              margin="normal"
            />
          </CardContent>
        </Card>

        {/* Banner Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Banner Section
            </Typography>
            <Grid container spacing={2}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Banner Title Heading"
                  value={formData.banner?.banner_title_heading || ""}
                  onChange={handleInputChange('banner', 'banner_title_heading', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Banner Text"
                  value={formData.banner?.banner_text || ""}
                  onChange={handleInputChange('banner', 'banner_text', null, null)}
                  margin="normal"
                  multiline
                  rows={3}
                />
              </Grid>
              <Grid item xs={12}>
                <Typography variant="subtitle2" gutterBottom>
                  Background Image
                </Typography>
                <input
                  type="file"
                  accept="image/*"
                  onChange={(e) => handleFileChange('banner', 'background_image_url', null, null)(e)}
                  style={{ width: '100%' }}
                />
                {isUploading('banner', 'background_image_url', null, null) && (
                  <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                    <CircularProgress size={20} />
                    <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                  </Box>
                )}
                {getUploadError('banner', 'background_image_url', null, null) && (
                  <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('banner', 'background_image_url', null, null)}</Alert>
                )}
                {formData.banner.background_image_url && (
                  <Alert severity="success" sx={{ mt: 1 }}>
                    <Typography variant="body2">
                      <strong>Uploaded:</strong> {formData.banner.background_image_url}
                    </Typography>
                  </Alert>
                )}
              </Grid>
            </Grid>
          </CardContent>
        </Card>

        {/* Services Tabs Section Image */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Services Tabs Section Image
            </Typography>
            <input
              type="file"
              accept="image/*"
              onChange={(e) => handleFileChange('services_tabs_section_image', null, null, null)(e)}
              style={{ width: '100%' }}
            />
            {isUploading('services_tabs_section_image', null, null, null) && (
              <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                <CircularProgress size={20} />
                <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
              </Box>
            )}
            {getUploadError('services_tabs_section_image', null, null, null) && (
              <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('services_tabs_section_image', null, null, null)}</Alert>
            )}
            {formData.services_tabs_section_image && (
              <Alert severity="success" sx={{ mt: 1 }}>
                <Typography variant="body2">
                  <strong>Uploaded:</strong> {formData.services_tabs_section_image}
                </Typography>
              </Alert>
            )}
          </CardContent>
        </Card>

        {/* Stand Out Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Stand Out Section
            </Typography>
            <Grid container spacing={2}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Heading"
                  value={formData.stand_out_section?.section_heading || ""}
                  onChange={handleInputChange('stand_out_section', 'section_heading', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Text"
                  value={formData.stand_out_section?.section_text || ""}
                  onChange={handleInputChange('stand_out_section', 'section_text', null, null)}
                  margin="normal"
                  multiline
                  rows={3}
                />
              </Grid>
            </Grid>
          </CardContent>
        </Card>

        {/* Why Us Image */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Why Us Image
            </Typography>
            <input
              type="file"
              accept="image/*"
              onChange={(e) => handleFileChange('why_us_image', null, null, null)(e)}
              style={{ width: '100%' }}
            />
            {isUploading('why_us_image', null, null, null) && (
              <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                <CircularProgress size={20} />
                <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
              </Box>
            )}
            {getUploadError('why_us_image', null, null, null) && (
              <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('why_us_image', null, null, null)}</Alert>
            )}
            {formData.why_us_image && (
              <Alert severity="success" sx={{ mt: 1 }}>
                <Typography variant="body2">
                  <strong>Uploaded:</strong> {formData.why_us_image}
                </Typography>
              </Alert>
            )}
          </CardContent>
        </Card>

        {/* Why Choose Us Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Why Choose Us Section
            </Typography>
            <TextField
              fullWidth
              label="Section Heading"
              value={formData.why_choose_us_section?.section_heading || ""}
              onChange={handleInputChange('why_choose_us_section', 'section_heading', null, null)}
              margin="normal"
              style={{ marginBottom: '16px' }}
            />
            <Stack spacing={3}>
              {formData.why_choose_us_section?.features?.map((feature, index) => (
                <Paper key={index} sx={{ p: 2 }}>
                  <Grid container spacing={2}>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Feature Title"
                        value={feature.title}
                        onChange={handleInputChange('why_choose_us_section', 'features', index, 'title')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Feature Text"
                        value={feature.text}
                        onChange={handleInputChange('why_choose_us_section', 'features', index, 'text')}
                        margin="normal"
                        multiline
                        rows={2}
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <Typography variant="subtitle2" gutterBottom>
                        Feature Image
                      </Typography>
                      <input
                        type="file"
                        accept="image/*"
                        onChange={handleFileChange('why_choose_us_section', 'features', index, 'image')}
                        style={{ width: '100%' }}
                      />
                      {isUploading('why_choose_us_section', 'features', index, 'image') && (
                        <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                          <CircularProgress size={20} />
                          <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                        </Box>
                      )}
                      {getUploadError('why_choose_us_section', 'features', index, 'image') && (
                        <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('why_choose_us_section', 'features', index, 'image')}</Alert>
                      )}
                      {feature.image && (
                        <Alert severity="success" sx={{ mt: 1 }}>
                          <Typography variant="body2">
                            <strong>Uploaded:</strong> {feature.image}
                          </Typography>
                        </Alert>
                      )}
                    </Grid>
                    <Grid item xs={12} md={12} style={{ display: 'flex', justifyContent: 'flex-end' }}>
                      <IconButton
                        onClick={() => removeFeature(index)}
                        color="error"
                        disabled={formData.why_choose_us_section?.features?.length === 1}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </Grid>
                  </Grid>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addFeature}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Feature
            </Button>
          </CardContent>
        </Card>

        {/* All Services Grid */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              All Services Grid
            </Typography>
            <TextField
              fullWidth
              label="Section Heading"
              value={formData.all_services_grid?.section_heading || ""}
              onChange={handleInputChange('all_services_grid', 'section_heading', null, null)}
              margin="normal"
              style={{ marginBottom: '16px' }}
            />
            <Stack spacing={2}>
              {formData.all_services_grid?.services?.map((service, index) => (
                <Box key={index} sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                  <TextField
                    fullWidth
                    label={`Service ${index + 1}`}
                    value={service}
                    onChange={handleInputChange('all_services_grid', 'services', index, null)}
                    margin="normal"
                  />
                  <IconButton
                    onClick={() => removeService(index)}
                    color="error"
                    disabled={formData.all_services_grid?.services?.length === 1}
                    style={{ display: 'flex', justifyContent: 'flex-end' }}
                  >
                    <DeleteIcon />
                  </IconButton>
                </Box>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addService}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Service
            </Button>
          </CardContent>
        </Card>

        {/* Our Services Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Our Services Section
            </Typography>
            <Grid container spacing={2} sx={{ mb: 3 }}>
              <Grid item xs={12} md={12}>
                <TextField
                  fullWidth
                  label="Section Heading"
                  value={formData.our_services_section?.section_heading || ""}
                  onChange={handleInputChange('our_services_section', 'section_heading', null, null)}
                  margin="normal"
                />
              </Grid>
              <Grid item xs={12} md={12}>
                <Typography variant="subtitle2" gutterBottom>
                  Section Background
                </Typography>
                <input
                  type="file"
                  accept="image/*"
                  onChange={(e) => handleFileChange('our_services_section', 'section_background', null, null)(e)}
                  style={{ width: '100%' }}
                />
                {isUploading('our_services_section', 'section_background', null, null) && (
                  <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                    <CircularProgress size={20} />
                    <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                  </Box>
                )}
                {getUploadError('our_services_section', 'section_background', null, null) && (
                  <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_services_section', 'section_background', null, null)}</Alert>
                )}
                {formData.our_services_section.section_background && (
                  <Alert severity="success" sx={{ mt: 1 }}>
                    <Typography variant="body2">
                      <strong>Uploaded:</strong> {formData.our_services_section.section_background}
                    </Typography>
                  </Alert>
                )}
              </Grid>
            </Grid>
            <Stack spacing={3}>
              {formData.our_services_section?.services?.map((service, index) => (
                <Paper key={index} sx={{ p: 2 }}>
                  <Grid container spacing={2}>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Service Title"
                        value={service.title}
                        onChange={handleInputChange('our_services_section', 'services', index, 'title')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Description"
                        value={service.description}
                        onChange={handleInputChange('our_services_section', 'services', index, 'description')}
                        margin="normal"
                        multiline
                        rows={2}
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <Typography variant="subtitle2" gutterBottom>
                        Service Image
                      </Typography>
                      <input
                        type="file"
                        accept="image/*"
                        onChange={handleFileChange('our_services_section', 'services', index, 'image')}
                        style={{ width: '100%' }}
                      />
                      {isUploading('our_services_section', 'services', index, 'image') && (
                        <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                          <CircularProgress size={20} />
                          <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                        </Box>
                      )}
                      {getUploadError('our_services_section', 'services', index, 'image') && (
                        <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('our_services_section', 'services', index, 'image')}</Alert>
                      )}
                      {service.image && (
                        <Alert severity="success" sx={{ mt: 1 }}>
                          <Typography variant="body2">
                            <strong>Uploaded:</strong> {service.image}
                          </Typography>
                        </Alert>
                      )}
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Button Text"
                        value={service.buttton_text}
                        onChange={handleInputChange('our_services_section', 'services', index, 'buttton_text')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12}>
                      <TextField
                        fullWidth
                        label="Button Link"
                        value={service.button_link}
                        onChange={handleInputChange('our_services_section', 'services', index, 'button_link')}
                        margin="normal"
                      />
                    </Grid>
                    <Grid item xs={12} md={12} style={{ display: 'flex', justifyContent: 'flex-end' }}>
                      <IconButton
                        onClick={() => removeOurService(index)}
                        color="error"
                        disabled={formData.our_services_section?.services?.length === 1}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </Grid>
                  </Grid>
                </Paper>
              ))}
            </Stack>
            <Button
              startIcon={<AddIcon />}
              onClick={addOurService}
              variant="outlined"
              sx={{ mt: 2 }}
            >
              Add Service
            </Button>
          </CardContent>
        </Card>

        {/* Page Settings Section */}
        <Card sx={{ mb: 3 }}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              Page Settings
            </Typography>
            <Box sx={{ display: 'flex', gap: 4, flexWrap: 'wrap' }}>
              <FormControlLabel
                control={
                  <Switch
                    checked={formData.enable_review}
                    onChange={handleSwitchChange('enable_review')}
                    color="primary"
                  />
                }
                label="Enable Review on this page"
              />
              <FormControlLabel
                control={
                  <Switch
                    checked={formData.enable_faq}
                    onChange={handleSwitchChange('enable_faq')}
                    color="primary"
                  />
                }
                label="Enable FAQ on this page"
              />
            </Box>
          </CardContent>
        </Card>

        {/* Submit Button */}
        <Box sx={{ display: 'flex', justifyContent: 'center', mt: 4 }}>
          <Button
            type="submit"
            variant="contained"
            size="large"
            sx={{ minWidth: 200 }}
          >
            {isEdit ? 'Update Page' : 'Submit Form'}
          </Button>
        </Box>
      </Box>
      )}
    </Container>
  );
};

export default BrandAgencyServiceForm;