import React, { useState, useEffect } from 'react';
import {
  Box,
  TextField,
  Button,
  Typography,
  Paper,
  Grid,
  Card,
  CardContent,
  IconButton,
  Divider,
  Container,
  Stack,
  CircularProgress,
  Alert,
  Switch,
  FormControlLabel
} from '@mui/material';
import { Add as AddIcon, Delete as DeleteIcon } from '@mui/icons-material';
import { useSelector, useDispatch } from 'react-redux';
import { GetAllServicesPages, SaveAppDevTemplatePage, UpdateAppDevTemplatePage, UploadServiceMediaAction } from '../../store/slices/servicesSlice/servicesSlice';
import { toast } from 'react-toastify';
import { GetAllServicesTabs } from '../../store/slices/propertyManagementSlice/propertyManagementSlice';
import { useModal } from '../../hooks/useModal';

const ApplicationDevServiceForm = ({ isEdit, initialData, onSuccess, handleModalClose, setIsEdit }) => {

  const { services } = useSelector(state => state.ServicesReducer);
  const dispatch = useDispatch();
  const [uploadingFiles, setUploadingFiles] = useState({});
  const [uploadErrors, setUploadErrors] = useState({});
  const { handleModalClose: defaultHandleModalClose } = useModal()

  // Use passed handleModalClose if available, otherwise use default
  const modalCloseHandler = handleModalClose || defaultHandleModalClose;

  console.log('[services]', services);

  const [formData, setFormData] = useState({
    page_heading: "",
    enable_review: false,
    enable_faq: false,
    banner: {
      banner_title_heading: "",
      banner_text: "",
      background_image_url: ""
    },
    services_tabs_section_image: "",
    services_tabs_section_text: "",
    our_services_section: {
      heading: "",
      text: ""
    },
    services_tab_section: {
      services: [
        {
          title: "",
          icon_url: "",
          description: "",
          button_text: "",
          button_link: ""
        }
      ]
    },
    all_services_grid: {
      Services_grid_text: "",
      services: [""]
    },
    info_cards_section: {
      section_heading: "",
      section_text: "",
      cards: [
        {
          heading: "",
          content: ""
        }
      ]
    }
  });

  // Pre-fill form data when in edit mode
  useEffect(() => {
    if (isEdit && initialData) {
      console.log('initialData', initialData);
      // Ensure all sections have proper structure even if they're null in the API response
      const processedData = {
        page_heading: initialData.page_heading || "",
        enable_review: initialData.enable_review || false,
        enable_faq: initialData.enable_faq || false,
        banner: {
          banner_title_heading: initialData.banner?.banner_title_heading || "",
          banner_text: initialData.banner?.banner_text || "",
          background_image_url: initialData.banner?.background_image_url || ""
        },
        services_tabs_section_image: initialData.services_tabs_section_image || "",
        services_tabs_section_text: initialData.services_tabs_section_text || "",
        our_services_section: {
          heading: initialData.our_services_section?.heading || "",
          text: initialData.our_services_section?.text || ""
        },
        services_tab_section: {
          services: initialData.services_tab_section?.services || [
            {
              title: "",
              icon_url: "",
              description: "",
              button_text: "",
              button_link: ""
            }
          ]
        },
        all_services_grid: {
          Services_grid_text: initialData.all_services_grid?.Services_grid_text || "",
          services: initialData.all_services_grid?.services || [""]
        },
        info_cards_section: {
          section_heading: initialData.info_cards_section?.section_heading || "",
          section_text: initialData.info_cards_section?.section_text || "",
          cards: initialData.info_cards_section?.cards || [
            {
              heading: "",
              content: ""
            }
          ]
        }
      };
      setFormData(processedData);
    }
  }, [isEdit, initialData]);

  // Debug: Log formData changes
  useEffect(() => {
    console.log('formData changed:', formData);
  }, [formData]);

  const handleSubmit = (event) => {
    event.preventDefault();
    console.log('Form Data:', formData);
    if (formData.page_heading) {
      if (isEdit) {
        console.log('initialData', initialData);
        console.log('formData', formData);
        
        dispatch(UpdateAppDevTemplatePage({ id: initialData.id, data: formData }))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Updated');
              modalCloseHandler();
              dispatch(GetAllServicesPages());
              if (onSuccess) onSuccess(response.data);
              if (setIsEdit) setIsEdit(false);
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          });
      } else {
        dispatch(SaveAppDevTemplatePage(formData))
          .unwrap()
          .then((response) => {
            console.log(response);
            if (response.success) {
              toast.success('Page Successfully Created');
              modalCloseHandler();
              dispatch(GetAllServicesPages());
              if (onSuccess) onSuccess(response.data);
            } else {
              toast.error(response.message);
            }
          })
          .catch((error) => {
            toast.error(error);
          });
      }
    } else {
      toast.error('Page Heading is required');
    }
  };

  const handleInputChange = (section, subsection, index, field) => (event) => {
    const value = event.target.value;
    
    if (section && !subsection && !index && !field) {
      // Handle direct field (like page_heading)
      setFormData(prev => ({
        ...prev,
        [section]: value
      }));
    } else if (section && subsection && index !== null && field) {
      // Handle nested array item field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? { ...item, [field]: value } : item
          )
        }
      }));
    } else if (section && subsection && index !== null) {
      // Handle array item (for services array)
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: prev[section][subsection].map((item, i) =>
            i === index ? value : item
          )
        }
      }));
    } else if (section && subsection) {
      // Handle nested object field
      setFormData(prev => ({
        ...prev,
        [section]: {
          ...prev[section],
          [subsection]: value
        }
      }));
    }
  };

  const handleSwitchChange = (field) => (event) => {
    setFormData(prev => ({
      ...prev,
      [field]: event.target.checked
    }));
  };

  const handleFileChange = (section, subsection, index, field) => async (event) => {
    const file = event.target.files[0];
    if (!file) return;

    // Create a unique key for this file upload
    const uploadKey = `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;

    // Set uploading state
    setUploadingFiles(prev => ({ ...prev, [uploadKey]: true }));
    setUploadErrors(prev => ({ ...prev, [uploadKey]: null }));

    try {
      const result = await dispatch(UploadServiceMediaAction({ file })).unwrap();

      if (result.success && result.data?.file_url) {
        // Update form data with the returned file_url
        if (section && subsection && index !== null && field) {
          // Handle nested array item field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: prev[section][subsection].map((item, i) =>
                i === index ? { ...item, [field]: result.data.file_url } : item
              )
            }
          }));
        } else if (section && subsection) {
          // Handle nested object field
          setFormData(prev => ({
            ...prev,
            [section]: {
              ...prev[section],
              [subsection]: result.data.file_url
            }
          }));
        } else if (section) {
          // Handle direct field
          setFormData(prev => ({
            ...prev,
            [section]: result.data.file_url
          }));
        }
      }
    } catch (error) {
      console.error('File upload failed:', error);
      setUploadErrors(prev => ({
        ...prev,
        [uploadKey]: error || 'File upload failed'
      }));
    } finally {
      setUploadingFiles(prev => ({ ...prev, [uploadKey]: false }));
    }
  };

  const getUploadKey = (section, subsection, index, field) => {
    return `${section}_${subsection || 'direct'}_${index !== null ? index : 'none'}_${field || 'none'}`;
  };

  const isUploading = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadingFiles[key] || false;
  };

  const getUploadError = (section, subsection, index, field) => {
    const key = getUploadKey(section, subsection, index, field);
    return uploadErrors[key] || null;
  };

  const addService = () => {
    setFormData(prev => ({
      ...prev,
      services_tab_section: {
        ...prev.services_tab_section,
        services: [...prev.services_tab_section.services, { title: "", icon_url: "", description: "", button_text: "", button_link: "" }]
      }
    }));
  };

  const removeService = (index) => {
    setFormData(prev => ({
      ...prev,
      services_tab_section: {
        ...prev.services_tab_section,
        services: prev.services_tab_section.services.filter((_, i) => i !== index)
      }
    }));
  };

  const addGridService = () => {
    setFormData(prev => ({
      ...prev,
      all_services_grid: {
        ...prev.all_services_grid,
        services: [...prev.all_services_grid.services, ""]
      }
    }));
  };

  const removeGridService = (index) => {
    setFormData(prev => ({
      ...prev,
      all_services_grid: {
        ...prev.all_services_grid,
        services: prev.all_services_grid.services.filter((_, i) => i !== index)
      }
    }));
  };

  const addCard = () => {
    setFormData(prev => ({
      ...prev,
      info_cards_section: {
        ...prev.info_cards_section,
        cards: [...prev.info_cards_section.cards, { heading: "", content: "" }]
      }
    }));
  };

  const removeCard = (index) => {
    setFormData(prev => ({
      ...prev,
      info_cards_section: {
        ...prev.info_cards_section,
        cards: prev.info_cards_section.cards.filter((_, i) => i !== index)
      }
    }));
  };


  return (
    <Container maxWidth="xl" sx={{ py: 4 }}>
      <Paper elevation={3} sx={{ p: 4 }}>

        <Box component="form" onSubmit={handleSubmit}>
          {/* Page Heading */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Heading
              </Typography>
              <TextField
                fullWidth
                label="Page Heading"
                value={formData.page_heading}
                onChange={(e) => handleInputChange('page_heading', null, null, null)(e)}
                placeholder="Enter page heading"
                variant="outlined"
                margin="normal"
              />
            </CardContent>
          </Card>

          {/* Banner Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Banner Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Title Heading"
                    value={formData.banner?.banner_title_heading || ""}
                    onChange={(e) => handleInputChange('banner', 'banner_title_heading', null, null)(e)}
                    placeholder="Enter banner title heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Banner Text"
                    value={formData.banner?.banner_text || ""}
                    onChange={(e) => handleInputChange('banner', 'banner_text', null, null)(e)}
                    placeholder="Enter banner text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12}>
                  <Typography variant="subtitle1" gutterBottom>
                    Background Image
                  </Typography>
                  <input
                    type="file"
                    accept="image/*"
                    onChange={(e) => handleFileChange('banner', 'background_image_url', null, null)(e)}
                    style={{ width: '100%' }}
                  />
                  {isUploading('banner', 'background_image_url', null, null) && (
                    <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                      <CircularProgress size={20} />
                      <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                    </Box>
                  )}
                  {getUploadError('banner', 'background_image_url', null, null) && (
                    <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('banner', 'background_image_url', null, null)}</Alert>
                  )}
                  {formData.banner?.background_image_url && (
                    <Alert severity="success" sx={{ mt: 1 }}>
                      <Typography variant="body2">
                        <strong>Uploaded:</strong> {formData.banner.background_image_url}
                      </Typography>
                    </Alert>
                  )}
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Services Tabs Section Image */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Services Tabs Section Image
              </Typography>
              <Typography variant="subtitle1" gutterBottom>
                Section Image
              </Typography>
              <input
                type="file"
                accept="image/*"
                onChange={(e) => handleFileChange('services_tabs_section_image', null, null, null)(e)}
                style={{ width: '100%' }}
              />
              {isUploading('services_tabs_section_image', null, null, null) && (
                <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                  <CircularProgress size={20} />
                  <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                </Box>
              )}
              {getUploadError('services_tabs_section_image', null, null, null) && (
                <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('services_tabs_section_image', null, null, null)}</Alert>
              )}
              {formData.services_tabs_section_image && (
                <Alert severity="success" sx={{ mt: 1 }}>
                  <Typography variant="body2">
                    <strong>Uploaded:</strong> {formData.services_tabs_section_image}
                  </Typography>
                </Alert>
              )}
              <TextField
                fullWidth
                label="Section Text"
                value={formData.services_tabs_section_text || ""}
                onChange={(e) => handleInputChange('services_tabs_section_text', null, null, null)(e)}
                placeholder="Enter section text"
                variant="outlined"
                multiline
                rows={3}
                margin="normal"
                sx={{ mt: 2 }}
              />
            </CardContent>
          </Card>

          {/* Our Services Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Our Services Section
              </Typography>
              <Grid container spacing={3}>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Heading"
                    value={formData.our_services_section?.heading || ""}
                    onChange={(e) => handleInputChange('our_services_section', 'heading', null, null)(e)}
                    placeholder="Enter section heading"
                    variant="outlined"
                    margin="normal"
                  />
                </Grid>
                <Grid item xs={12} md={12}>
                  <TextField
                    fullWidth
                    label="Section Text"
                    value={formData.our_services_section?.text || ""}
                    onChange={(e) => handleInputChange('our_services_section', 'text', null, null)(e)}
                    placeholder="Enter section text"
                    variant="outlined"
                    multiline
                    rows={3}
                    margin="normal"
                  />
                </Grid>
              </Grid>
            </CardContent>
          </Card>

          {/* Services Tab Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Services Tab Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addService}

                >
                  Add Service
                </Button>
              </Box>
              <Stack spacing={3}>
                {formData.services_tab_section?.services?.map((service, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Service {index + 1}</Typography>
                      {formData.services_tab_section?.services?.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeService(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Title"
                          value={service.title}
                          onChange={(e) => handleInputChange('services_tab_section', 'services', index, 'title')(e)}
                          placeholder="Enter service title"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12}>
                        <Typography variant="subtitle2" gutterBottom>
                          Icon Image
                        </Typography>
                        <input
                          type="file"
                          accept="image/*"
                          onChange={(e) => handleFileChange('services_tab_section', 'services', index, 'icon_url')(e)}
                          style={{ width: '100%' }}
                        />
                        {isUploading('services_tab_section', 'services', index, 'icon_url') && (
                          <Box sx={{ mt: 2, display: 'flex', alignItems: 'center' }}>
                            <CircularProgress size={20} />
                            <Typography variant="body2" sx={{ ml: 1 }}>Uploading...</Typography>
                          </Box>
                        )}
                        {getUploadError('services_tab_section', 'services', index, 'icon_url') && (
                          <Alert severity="error" sx={{ mt: 1 }}>{getUploadError('services_tab_section', 'services', index, 'icon_url')}</Alert>
                        )}
                        {service.icon_url && (
                          <Alert severity="success" sx={{ mt: 1 }}>
                            <Typography variant="body2">
                              <strong>Uploaded:</strong> {service.icon_url}
                            </Typography>
                          </Alert>
                        )}
                      </Grid>
                      <Grid item xs={12} md={12}>
                        <TextField
                          fullWidth
                          label="Description"
                          value={service.description}
                          onChange={(e) => handleInputChange('services_tab_section', 'services', index, 'description')(e)}
                          placeholder="Enter service description"
                          variant="outlined"
                          multiline
                          rows={3}
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Button Text"
                          value={service.button_text}
                          onChange={(e) => handleInputChange('services_tab_section', 'services', index, 'button_text')(e)}
                          placeholder="Enter button text"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12} md={6}>
                        <TextField
                          fullWidth
                          label="Button Link"
                          value={service.button_link}
                          onChange={(e) => handleInputChange('services_tab_section', 'services', index, 'button_link')(e)}
                          placeholder="Enter button link"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* All Services Grid */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  All Services Grid
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addGridService}

                >
                  Add Service
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Text"
                value={formData.all_services_grid.Services_grid_text}
                onChange={(e) => handleInputChange('all_services_grid', 'Services_grid_text', null, null)(e)}
                placeholder="Enter section text"
                variant="outlined"
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={2}>
                {formData.all_services_grid.services.map((service, index) => (
                  <Box key={index} display="flex" gap={2} alignItems="center">
                    <TextField
                      fullWidth
                      label={`Service ${index + 1}`}
                      value={service}
                      onChange={(e) => handleInputChange('all_services_grid', 'services', index, null)(e)}
                      placeholder="Enter service name"
                      variant="outlined"
                      size="small"
                    />
                    {formData.all_services_grid.services.length > 1 && (
                      <IconButton
                        color="error"
                        onClick={() => removeGridService(index)}
                        size="small"
                      >
                        <DeleteIcon />
                      </IconButton>
                    )}
                  </Box>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Info Cards Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                <Typography variant="h5" >
                  Info Cards Section
                </Typography>
                <Button
                  variant="contained"
                  startIcon={<AddIcon />}
                  onClick={addCard}

                >
                  Add Card
                </Button>
              </Box>
              <TextField
                fullWidth
                label="Section Heading"
                value={formData.info_cards_section.section_heading}
                onChange={(e) => handleInputChange('info_cards_section', 'section_heading', null, null)(e)}
                placeholder="Enter section heading"
                variant="outlined"
                margin="normal"
                sx={{ mb: 2 }}
              />
              <TextField
                fullWidth
                label="Section Text"
                value={formData.info_cards_section.section_text}
                onChange={(e) => handleInputChange('info_cards_section', 'section_text', null, null)(e)}
                placeholder="Enter section text"
                variant="outlined"
                multiline
                rows={3}
                margin="normal"
                sx={{ mb: 3 }}
              />
              <Stack spacing={3}>
                {formData.info_cards_section.cards.map((card, index) => (
                  <Card key={index} variant="outlined" sx={{ p: 2, bgcolor: 'grey.50' }}>
                    <Box display="flex" justifyContent="space-between" alignItems="center" mb={2}>
                      <Typography variant="h6">Card {index + 1}</Typography>
                      {formData.info_cards_section.cards.length > 1 && (
                        <IconButton
                          color="error"
                          onClick={() => removeCard(index)}
                          size="small"
                        >
                          <DeleteIcon />
                        </IconButton>
                      )}
                    </Box>
                    <Grid container spacing={2}>
                      <Grid item xs={12}>
                        <TextField
                          fullWidth
                          label="Heading"
                          value={card.heading}
                          onChange={(e) => handleInputChange('info_cards_section', 'cards', index, 'heading')(e)}
                          placeholder="Enter card heading"
                          variant="outlined"
                          size="small"
                        />
                      </Grid>
                      <Grid item xs={12}>
                        <TextField
                          fullWidth
                          label="Content"
                          value={card.content}
                          onChange={(e) => handleInputChange('info_cards_section', 'cards', index, 'content')(e)}
                          placeholder="Enter card content"
                          variant="outlined"
                          multiline
                          rows={4}
                          size="small"
                        />
                      </Grid>
                    </Grid>
                  </Card>
                ))}
              </Stack>
            </CardContent>
          </Card>

          {/* Page Settings Section */}
          <Card sx={{ mb: 4 }}>
            <CardContent>
              <Typography variant="h5" gutterBottom >
                Page Settings
              </Typography>
              <Box sx={{ display: 'flex', gap: 4, flexWrap: 'wrap' }}>
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_review}
                      onChange={handleSwitchChange('enable_review')}
                      color="primary"
                    />
                  }
                  label="Enable Review on this page"
                />
                <FormControlLabel
                  control={
                    <Switch
                      checked={formData.enable_faq}
                      onChange={handleSwitchChange('enable_faq')}
                      color="primary"
                    />
                  }
                  label="Enable FAQ on this page"
                />
              </Box>
            </CardContent>
          </Card>

          {/* Submit Button */}
          <Box display="flex" justifyContent="center" mt={4}>
            <Button
              type="submit"
              variant="contained"
              size="large"

              sx={{ py: 1, px: 4, fontSize: '1rem' }}
            >
              {isEdit ? 'Update Page' : 'Submit Form'}
            </Button>
          </Box>
        </Box>
      </Paper>
    </Container>
  );
};

export default ApplicationDevServiceForm;